;; --------------------------------------------------------------------
;; HybridSAL
;; Copyright (C) 2006, SRI International.  All Rights Reserved.
;; 
;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License
;; as published by the Free Software Foundation; either version 2
;; of the License, or (at your option) any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
;; --------------------------------------------------------------------

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; unparse.lisp -- 
;; Author          : Ashish Tiwari
;; Created On      : Thu May 31 2007
;; Update Count    : 0
;; 
;; HISTORY
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package :sal)

(defvar *pretty-printing-decl-list* nil)
(defvar *pretty-printed-prefix* nil)

;;; Unparse takes the following keywords:
;;; :string :stream :char-width :file :style
(defun unparse (obj &key string stream file char-width
		    length level lines (pretty t))
   (let ((*print-length* length)
	 (*print-level* level)
	 (*print-lines* lines)
	 (*print-pretty* pretty)
	 (*print-escape* nil)
	 (*print-readably* nil)
	 (*print-right-margin* (or char-width 72)))
     (cond (string
	    (decf *print-right-margin* 4)
	    (with-output-to-string (*standard-output*)
	      (pp obj)))
	   (stream
	    (let ((*standard-output* stream))
	      (pp obj)))
	   (file
	    (with-open-file (*standard-output* file :direction :output)
	      (pp obj)))
	   (t (decf *print-right-margin* 4)
	      (pp obj)))))

(defun pp (obj)
  (let ((*pretty-printing-decl-list* nil)
	(*pretty-printed-prefix* nil))
    (pprint-logical-block (nil nil)
      (pp* obj))))

(defmethod pp* ((ctx context))
  (pprint-logical-block (nil nil)
    (write (id ctx))
    (pp-context-formals (formals ctx))
    (write ": ")
    (pprint-indent :block 2)
    (write 'CONTEXT)
    (write-char #\space)
    (write '=)
    (pprint-indent :block 1)
    (pprint-newline :mandatory)
    (write 'BEGIN)
    (pprint-indent :block 2)
    (pprint-newline :mandatory)
    (pp-context-body (context-body ctx))
    (pprint-indent :block 1)
    (pprint-newline :mandatory)
    (write 'END)))

(defun pp-context-body (decls)
  (pprint-logical-block (nil decls)
    (loop (pp* (pprint-pop))
	  (write ";")
	  (pprint-exit-if-list-exhausted)
	  (pprint-newline :mandatory)
	  (pprint-newline :mandatory))))
	  

(defun pp-context-formals (formals)
  (pp-formals formals "{" "}"))

(defun pp-formals (formals pre suf)
  (when formals
    (pprint-logical-block (nil formals :prefix pre :suffix suf)
      (loop (let ((elt (pprint-pop)))
	      (pp* elt)
	      (pprint-exit-if-list-exhausted)
	      (write-char #\,)
	      (write-char #\space)
	      (pprint-newline :fill))))))

;; --------------------------------------------------------
;; sal-declaration: 1 around method + individual methods
;; --------------------------------------------------------
(defmethod pp* :around ((decl sal-declaration))
  (with-slots (id formals chain? semi) decl
    (cond ((and chain?
		*pretty-printing-decl-list*)
	   (write id)
	   (write-char #\,)
	   (write-char #\space)
	   (pprint-newline :fill))
	  (t (write id)
	     (pp-formals formals "(" ")")
	     (write-char #\:)
	     (write-char #\space)
	     (if (module-declaration? decl)
	         (pp-module-parameters (parameters decl)))
	     (pprint-newline :fill)
	     (call-next-method)
	     (when semi (write-char #\;))))))

(defmethod pp* ((decl var-decl))
  (pp* (declared-type decl)))

(defmethod pp* ((decl constant-declaration))
  (pp* (declared-type decl))
  (write-char #\space)
  (write #\=)
  (pprint-indent :block 2)
  (write-char #\space)
  (pprint-newline :fill)
  (pp* (definition decl))
  (pprint-indent :block 0))

(defmethod pp* ((decl type-declaration))
  (write 'TYPE)
  (when (type-expr decl)
    (write-char #\space)
    (write '=)
    (write-char #\space)
    (pp* (type-expr decl))))

(defmethod pp* ((decl assertion-declaration))
  (write (assertion-form decl))
  (pprint-indent :block 2)
  (write-char #\space)
  (pprint-newline :fill)
  (pp* (assertion decl))
  (pprint-indent :block 0))

(defmethod pp* ((decl module-declaration))
  (pprint-newline :fill)
  (write 'MODULE)
  (write-char #\space)
  (write #\=)
  (pprint-indent :block 2)
  (write-char #\space)
  (pprint-newline :fill)
  (pp* (module decl))
  (pprint-indent :block 0))

(defmethod pp* ((decl module-parameter-decl))
  (with-slots (declared-type) decl
    (pprint-newline :fill)
    (pp* declared-type)))

(defmethod pp* ((decl state-var-decl))
  (pp* (declared-type decl)))

(defun pp-module-parameters (parameters)
  (if parameters
      (sal-error parameters "Missing Code.~%")))
  ;; (pprint-logical-block (nil nil :prefix "[" :suffix "]")
    ;; (pp-base-chained-decls parameters))
;; --------------------------------------------------------

;; --------------------------------------------------------
;; Base Module
;; --------------------------------------------------------
(defmethod pp* ((m base-module))
  (write 'BEGIN)
  (pprint-newline :mandatory)
  (pp-base-declarations (declarations m))
  (pprint-newline :mandatory)
  (write 'END))

(defun pp-base-declarations (decls)
  (let ((*pretty-printing-decl-list* t))
    (pprint-logical-block (nil decls)
      (loop (pp-base-decl (pprint-pop))
	    (pprint-exit-if-list-exhausted)
	    (pprint-newline :mandatory)))))

(defun pp-base-decl (decl)
  (write (typecase decl
	   (input-decl 'INPUT)
	   (output-decl 'OUTPUT)
	   (global-decl 'GLOBAL)
	   (local-decl 'LOCAL)
	   (init-decl 'INITIALIZATION)
	   (trans-decl 'TRANSITION)
	   (def-decl 'DEFINITION)
	   (initfor-decl 'INITFORMULA)
	   (invar-decl 'INVARIANT)
	   (t (break))))
  (pprint-newline :mandatory)
  (pprint-indent :block 2)
  (pp* decl)
  (pprint-indent :block 0))
;; --------------------------------------------------------

;; --------------------------------------------------------
(defmethod pp* ((decl init-decl))
  (pprint-logical-block (nil nil)
    (pp-definitions-or-commands (definitions-or-commands decl))))

(defmethod pp* ((decl trans-decl))
  (pprint-logical-block (nil nil)
    (pp-definitions-or-commands (definitions-or-commands decl))))

(defun pp-definitions-or-commands (defs-or-cmds)
  (pprint-logical-block (nil (pp-collect-commands defs-or-cmds))
    (loop (let ((def-or-cmd (pprint-pop)))
	    (if (listp def-or-cmd)
		(pp-command def-or-cmd)
		(pp* def-or-cmd)))
	  (pprint-exit-if-list-exhausted)
	  (write "; ")
	  (pprint-newline :fill))))

(defun pp-collect-commands (defs-or-cmds &optional cmds result)
  (if (null defs-or-cmds)
      (if cmds
	  (nreverse (cons (nreverse cmds) result))
	  (nreverse result))
      (if (guarded-command? (car defs-or-cmds))
	  (if (last-assignment? (car defs-or-cmds))
	      (pp-collect-commands
	       (cdr defs-or-cmds)
	       nil
	       (cons (nreverse (cons (car defs-or-cmds) cmds)) result))
	      (pp-collect-commands (cdr defs-or-cmds)
				   (cons (car defs-or-cmds) cmds)
				   result))
	  (pp-collect-commands (cdr defs-or-cmds)
			       nil
			       (cons (car defs-or-cmds)
				     (if cmds
					 (cons (nreverse cmds) result)
					 result))))))

(defun pp-command (cmd-list)
  (pprint-logical-block (nil cmd-list :prefix "[" :suffix "]")
    (loop (pp* (pprint-pop))
	  (pprint-newline :mandatory)
	  (pprint-exit-if-list-exhausted)
	  (write '[])
	  (pprint-newline :mandatory))))

(defmethod pp* ((cmd guarded-command))
  (pprint-logical-block (nil nil)
    (pp* (guard cmd))
    (pprint-newline :linear)
    (write-char #\space)
    (write '-->)
    (write-char #\space)
    (pprint-newline :linear)
    (pp-assignments (assignments cmd))))

(defun pp-assignments (assignments)
  (pprint-logical-block (nil assignments)
    (loop (pp* (pprint-pop))
	  (pprint-exit-if-list-exhausted)
	  (write ";")
	  (write-char #\space)
	  (pprint-newline :fill))))

(defmethod pp* ((g guard))
  (pp* (expression g)))

(defmethod pp* ((def simple-definition))
  (pp* (lhs def))
  (write-char #\space)
  (if (rhs-selection? (rhs-definition def))
      (write 'IN)
      (write '=))
  (write-char #\space)
  (pp* (rhs-definition def)))

(defmethod pp* ((ex rhs-selection))
  (pp* (expression ex)))

(defmethod pp* ((ex set-list-expr))
  (pprint-logical-block (nil (expressions ex) :prefix "{" :suffix "}")
    (loop (pp* (pprint-pop))
	  (pprint-exit-if-list-exhausted)
	  (write ", ")
	  (pprint-newline :fill))))

(defmethod pp* ((def def-decl))
  (pprint-logical-block (nil (definitions def))
    (loop (pp* (pprint-pop))
	  (pprint-exit-if-list-exhausted)
	  (write ";")
	  (pprint-newline :mandatory))))


(defmethod pp* ((te scalar-type))
  (pprint-logical-block (nil (identifiers te) :prefix "{" :suffix "}")
    (loop (write (pprint-pop))
	  (pprint-exit-if-list-exhausted)
	  (write ", ")
	  (pprint-newline :fill))))

(defmethod pp* ((ex next-operator))
  (pp* (name ex))
  (write "'"))

(defmethod pp* ((ex chained-conditional))
  (pprint-logical-block (nil nil)
    (pprint-indent :current 0)	;; changed from 2 to 0
    (write 'ELSIF)
    (write-char #\space)
    (pp* (nth 0 (exprs (argument ex))))
    (write-char #\space)
    (pprint-newline :linear)
    (write 'THEN)
    (write-char #\space)
    (pp* (nth 1 (exprs (argument ex))))
    (write-char #\space)
    (pprint-indent :block 0)
    (pprint-newline :linear)
    (cond ((chained-conditional? (nth 2 (exprs (argument ex))))
	   ;(write 'ELSIF)
	   (pp* (nth 2 (exprs (argument ex)))))
	   ;;(pp-chained-if-expr (else-part ex) nil)
	  (t (write 'ELSE)
	     (write-char #\space)
	     (pp* (nth 2 (exprs (argument ex))))))))

(defmethod pp* ((ex conditional))
  (pprint-logical-block (nil nil)
    (pprint-indent :current 0)	;; changed from 2 to 0
    (write 'IF)
    (write-char #\space)
    (pp* (nth 0 (exprs (argument ex))))
    (write-char #\space)
    (pprint-newline :linear)
    (write 'THEN)
    (write-char #\space)
    (pp* (nth 1 (exprs (argument ex))))
    (write-char #\space)
    (pprint-indent :block 0)
    (pprint-newline :linear)
    (cond ((chained-conditional? (nth 2 (exprs (argument ex))))
	   ;; (write 'ELSIF)
	   (pp* (nth 2 (exprs (argument ex)))))
	   ;;(pp-chained-if-expr (else-part ex) nil)
	  (t (write 'ELSE)
	     (write-char #\space)
	     (pp* (nth 2 (exprs (argument ex))))))
    (write-char #\space)
    (pprint-newline :linear)
    (write 'ENDIF)))

(defmethod pp* ((m asynchronous-composition))
  (pprint-logical-block (nil nil)
    (pp* (module1 m))
    (write-char #\space)
    (write-string "[]")
    (pprint-newline :fill)
    (write-char #\space)
    (pp* (module2 m))))

(defmethod pp* ((m synchronous-composition))
  (pprint-logical-block (nil nil)
    (pp* (module1 m))
    (write-char #\space)
    (write-string "||")
    (pprint-newline :fill)
    (write-char #\space)
    (pp* (module2 m))))

(defmethod pp* ((m observe-module))
  (pprint-logical-block (nil nil)
    (write 'OBSERVE)
    (pprint-newline :fill)
    (write-char #\space)
    (pp* (module1 m))
    (write-char #\space)
    (write 'WITH)
    (pprint-newline :fill)
    (write-char #\space)
    (pp* (module2 m))))

(defmethod pp* ((ass module-models))
  (pprint-logical-block (nil nil)
    (pp* (module ass))
    (write " |- " )
    (pp* (assertion ass))))

(defmethod pp* ((minst module-instance))
  (pprint-logical-block (nil nil)
    (pp* (mod-name minst))))

;;; Type expressions

(defmethod pp* ((te subrange))
  (pprint-logical-block (nil nil :prefix "[" :suffix "]")
    (pp* (lower-bound te))
    (write " .. ")
    (pp* (upper-bound te))))

(defmethod pp* ((te array-type))
  (pprint-logical-block (nil nil)
    (write "ARRAY ")
    (pp* (domain te))
    (pprint-indent :block 2)
    (pprint-newline :fill)
    (write " OF ")
    (pp* (range te))))

;;; Expressions
(defmethod pp* ((mn module-name))
  (pprint-logical-block (nil nil)
    (write (id mn))))

(defmethod pp* ((ex array-selection))
  (pprint-logical-block (nil nil)
    (pp* (operator ex))
    (write "[")
    (pp* (argument ex))
    (write "]")))

(defmethod pp* ((ex record-selection))
  (pprint-logical-block (nil nil)
    (pp* (record ex))
    (write ".")
    (pp* (field ex))))

(defmethod pp* ((ex tuple-selection))
  (pprint-logical-block (nil nil)
    (pp* (tuple ex))
    (write ".")
    (pp* (index ex))))

(defmethod pp* ((ex infix-application))
  (declare (special *infix-operator-precs*))
  (with-slots (operator argument) ex
    (if (and (typep operator '(and name-expr (not qualified-name-expr)))
	     (typep argument 'tuple-expr)
	     (= (length (exprs argument)) 2))
	(let ((lhs (car (exprs argument)))
	      (rhs (cadr (exprs argument)))
	      (oper (id operator)))
	  (pprint-logical-block (nil nil)
	    (pprint-indent :block 1)
	    (if (< (precedence lhs 'left)
		   (second (or (assoc oper *infix-operator-precs*)
			       (assoc 'IDENTIFIER *infix-operator-precs*))))
		(progn (write-char #\()
		       (pp* lhs)
		       (write-char #\)))
		(pp* lhs))
	    (write-char #\space)
	    (pprint-newline :fill)
	    (write (id operator))
	    (write-char #\space)
	    (pprint-newline :fill)
	    (if (< (precedence rhs 'right)
		   (third (or (assoc oper *infix-operator-precs*)
			       (assoc 'IDENTIFIER *infix-operator-precs*))))
		(progn (write-char #\()
		       (pp* rhs)
		       (write-char #\)))
		(pp* rhs))))
	(call-next-method))))

(defmethod pp* ((ex unary-application))
  (declare (special *unary-operator-precs*))
  (with-slots (operator argument) ex
    (if (typep operator '(and name-expr (not qualified-name-expr)))
	(pprint-logical-block (nil nil)
	  (pprint-indent :current 2)
	  (write (id operator))
	  (when (alpha-char-p (char (string (id operator)) 0))
	    (write-char #\space)
	    (pprint-newline :miser))
	  (if (>= (precedence argument 'right)
		  (second (assoc (id operator) *unary-operator-precs*)))
	      (pp* argument)
	      (pprint-logical-block (nil nil :prefix "(" :suffix ")")
		(pp* argument))))
	(call-next-method))))

(defmethod pp* ((ex application))
  (let ((operator (operator ex))
	(args (argument ex)))
    (pprint-logical-block (nil nil)
      (pprint-indent :current 2)
      (pp* operator) 
      (pprint-indent :block 4)
      (pp* args))))

(defmethod pp* ((arg tuple-literal))
  (pprint-logical-block (nil (exprs arg) :prefix "(" :suffix ")")
    (loop (pp* (pprint-pop))
	  (pprint-exit-if-list-exhausted)
	  (write ", ")
	  (pprint-newline :fill))))

(defmethod pp* ((ex qualified-name-expr))
  (pprint-logical-block (nil nil)
    (pp* (context-name ex))
    (pprint-newline :fill)
    (write "!")
    (write (id ex))))

(defmethod pp* ((ex name-expr))
  (write (id ex)))

(defmethod pp* ((ex name))
  (write (id ex)))

(defmethod pp* ((ex context-name))
  (write (id ex))
  (pp-actuals (actuals ex)))

(defmethod pp* ((ss symbol))
  (write ss)) ;; (break)

(defmethod pp* :around ((ex expression))
  (progn (dotimes (p (parens ex))
	   (write-char #\())
   	 (call-next-method)
	 (dotimes (p (parens ex))
	   (write-char #\)))))

(defmethod pp* ((ex numeral))
  (write (this-number ex)))

(defmethod pp* ((ff funtype))
  (write "[")
  (pp* (domain ff))
  (write " -> ")
  (pp* (range ff))
  (write "]"))
  
(defmethod pp-actuals (actuals)
  (let ((last-type (find-if #'type-expression? actuals
			    :from-end t :key #'expr)))
    (pprint-logical-block (nil actuals :prefix "{" :suffix "}")
      (unless last-type
	(write "; "))
      (loop (let ((actual (pprint-pop)))
	      (pp* actual)
	      (when (and last-type (eq actual last-type))
		(write ";"))
	      (pprint-exit-if-list-exhausted)
	      (unless (and last-type (eq actual last-type))
		(write ","))
	      (write-char #\space)
	      (pprint-newline :fill))))))

;;; Find the precedence of an expression.  For a given expression and ctx
;;; of left or right, determine the precedence number.

(defvar *infix-operators* '(<=> => OR XOR AND = /= > >= < <= + - * /))

(defvar *unary-operators* '(NOT -))

(defvar *unary-operator-precs*
  '((NOT 45)
    (-   95)))

(defparameter *infix-operator-precs*
  '((<=> 11 10)
    (=>  20 21)
    (OR  31 30)
    (XOR 31 30)
    (AND 40 41)
    (=   51 50)
    (/=  51 50)
    (>   61 60)
    (>=  61 60)
    (<   61 60)
    (<=  61 60)
    (IDENTIFIER 71 70)
    (+   81 80)
    (-   81 80)
    (*   91 90)
    (/   91 90)))

(defmethod precedence ((ex expr) ctx)
  (declare (ignore ctx))
  most-positive-fixnum)

(defmethod precedence :around ((expr expression) ctx)
  (declare (ignore ctx))
  (if (plusp (parens expr))
      most-positive-fixnum
      (call-next-method)))

;; Most types of expressions cannot be ambiguous (e.g. tuples, if-exprs).
(defmethod precedence ((expr expression) ctx)
  (declare (ignore ctx))
  most-positive-fixnum)

(defmethod precedence ((expr unary-application) ctx)
  (declare (special *unary-operator-precs*))
  (let ((precs (when (typep (operator expr) 'name-expr)
		 (assoc (id (operator expr)) *unary-operator-precs*))))
    (if precs
	(case ctx
	  (left (cadr precs))
	  (right most-positive-fixnum))
	(call-next-method))))

(defmethod precedence ((expr application) ctx)
  (case ctx
    (left most-positive-fixnum)
    (right most-positive-fixnum)))

(defmethod precedence ((expr infix-application) ctx)
  (if (and (typep (operator expr) 'name-expr)
	   (typep (argument expr) 'tuple-expr)
	   (= (length (exprs (argument expr))) 2))
      (let ((prec (or (assoc (id (operator expr)) *infix-operator-precs*)
		      (assoc 'IDENTIFIER *infix-operator-precs*))))
	(case ctx
	  (left (min (third prec)
		     (if (not (zerop (parens (cadr (exprs (argument expr))))))
			 most-positive-fixnum
			 (precedence (second (exprs (argument expr))) 'left))))
	  (right (second prec))))
      (call-next-method)))

(defmethod precedence ((expr name-expr) ctx)
  (declare (special *unary-operator-precs*))
  (if (and (eq ctx 'left)
	   (assoc (id expr) *unary-operator-precs*))
      0
      (call-next-method)))

;; --------------------------------------------------------
;; print-object
;; --------------------------------------------------------
(defmethod print-object ((obj sal-syntax) str)
  (unparse obj :stream str))
;(defmethod print-object ((var name-expr) stream)
  ;(format stream "~a" (id var)))
;; --------------------------------------------------------
