;; --------------------------------------------------------------------
;; HybridSAL
;; Copyright (C) 2006, SRI International.  All Rights Reserved.
;; 
;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License
;; as published by the Free Software Foundation; either version 2
;; of the License, or (at your option) any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
;; --------------------------------------------------------------------

;;;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; sal-class-definitions.lisp --
;; Author          : Ashish Tiwari
;; HISTORY
;; 05.29.07: reflecting changes in class-definitions.lisp
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package :sal)

;; -----------------------------------------------------
;; Generic
;; -----------------------------------------------------
(defmacro makesym (ctl &rest args)
   `(intern (format nil ,ctl ,@args) :sal))
   ;; `(intern "dummy" :sal)

(defmacro make-sal-instance (class attrs &rest initargs)
   `(apply #'make-instance ,class ,@initargs ,attrs))
;; -----------------------------------------------------

;; -----------------------------------------------------
;; MAIN
;; -----------------------------------------------------
(defun sal-parse (filename)
  (let ((sal-file (make-pathname :defaults filename :type "sal"))
	(xml-file (make-pathname :defaults filename :type "xml")))
    (cond ((probe-file sal-file)
	   (unless (and (probe-file xml-file)
			(<= (file-write-date sal-file)
			    (file-write-date xml-file)))
	     (excl:shell (format nil "~a/hybridsal2xml/hybridsal2xml ~a > ~a"
			   *sal-path*
			   (namestring sal-file) (namestring xml-file))))
	   (require :pxml)	;; Using (net.xml.parser:parse-xml)
	   (with-open-file (p xml-file)
	     (let ((lxml (net.xml.parser:parse-xml p :content-only p)))
	       (translate-xsal-to-sal lxml))))
	  ((probe-file xml-file)
	   (require :pxml)	;; Using (net.xml.parser:parse-xml)
	   (with-open-file (p xml-file)
	     (let ((lxml (net.xml.parser:parse-xml p :content-only p)))
	       (translate-xsal-to-sal lxml))))
	  (t (sal-error filename "File not found")))))

(defmethod sal-parse-term ((input string) &key (nt 'expression))
  (let ((cmd
	     (vector (format nil "~a/hybridsal2xml/hybridsal2xml" *sal-path*)
		     (format nil "~a/hybridsal2xml/hybridsal2xml" *sal-path*)
		     "-nt" (string nt) "-")))
    (multiple-value-bind (in out err pid)
	(excl:run-shell-command cmd :input :stream :output :stream
				:separate-streams t :wait nil)
      (declare (ignore err))
      (require :pxml)		;; Using (net.xml.parser:parse-xml)
      (format in "~a" input)
      (close in)
      (let ((lxml (net.xml.parser:parse-xml out :content-only out)))
	(close out)
	(sys:reap-os-subprocess :pid pid)
	(translate-xsal-to-sal lxml)))))
;; -----------------------------------------------------

;; -----------------------------------------------------
;; Removing empty elements from the lxml
;; -----------------------------------------------------
(defun remove-empty-text-elements (lxml)
  (cond ((null lxml) nil)
	((consp lxml)
	 (mapcar #'remove-empty-text-elements
	   (remove-if #'empty-text-element? lxml)))
	(t lxml)))

(defun empty-text-element? (elt)
  (and (stringp elt)
       (every #'whitespace-char? elt)))

(defun whitespace-char? (c)
  (member c '(#\space #\tab #\newline #\page #\return #\linefeed)))
;; -----------------------------------------------------

;; -----------------------------------------------------
;; xsal to class structure...
;; -----------------------------------------------------
(defun translate-xsal-to-sal (lxml)
  (assert (null (cdr lxml)))
  (translate-xsal-to-sal* (remove-empty-text-elements (car lxml))))

(defun translate-xsal-to-sal* (lxml)
  (cond ((consp lxml)
	 (let ((fun (element-to-sal-instance (car lxml))))
	   (apply fun
	     (translate-xsal-attributes (car lxml))
	     (mapcar #'translate-xsal-to-sal* (cdr lxml)))))
	((stringp lxml)
	 (intern lxml))
	(t (break "translate-xsal-to-sal*: Not a list or string"))))

(defun element-to-sal-instance (lxml)
  (let ((eltname (if (listp lxml)
		     (if (listp (car lxml)) (caar lxml) (car lxml))
		     lxml)))
    (assert (symbolp eltname))
    (makesym "mk-sal-~(~a~)" eltname)))
;; -----------------------------------------------------

;; -----------------------------------------------------
;; translate-xsal-attributes
;; -----------------------------------------------------
(defun translate-xsal-attributes (elt)
  (when (listp elt)
    (translate-xsal-attributes* (cdr elt) nil)))

(defun translate-xsal-attributes* (list attrs)
  (if (null list)
      attrs
      (translate-xsal-attributes*
       (cddr list)
       (nconc attrs
	      (translate-xsal-attribute (car list) (cadr list))))))

(defun translate-xsal-attribute (attr value)
  (case attr
    (PLACE (list 'place (translate-xsal-place value)))
    (INFIX (list 'infix? (if (string-equal value "YES") t nil)))
    (UNARY (list 'unary? (if (string-equal value "YES") t nil)))
    (CHAIN (list 'chain? (if (string-equal value "YES") t nil)))
    (ELSIF (list 'elsif? (if (string-equal value "YES") t nil)))
    (PARENS (list 'parens (parse-integer value :junk-allowed t)))
    (t (break "translate-xsal-attribute: ~a not handled" attr))))

(defun translate-xsal-place (string &optional (pos 0) place)
  (multiple-value-bind (num npos)
      (parse-integer string :start pos :junk-allowed t)
    (if (null num)
	(nreverse place)
	(translate-xsal-place string npos (cons num place)))))
;; -----------------------------------------------------
       
;; -----------------------------------------------------
;; mk-sal-<elementname>
;; -----------------------------------------------------
;(defun dummy (&rest p1)
  ;nil)
(defun mk-sal-context (attrs identifier parameters body)
  (make-sal-instance 'context attrs
    'id (id identifier)
    'formals (append (when (type-decls parameters)
		       (type-decls parameters))
		     (when (var-decls parameters)
		       (var-decls parameters)))
    'context-body (declarations body)))

(defun mk-sal-parameters (attrs &optional typedecls vardecls)
  (make-sal-instance 'parameters attrs
    'type-decls (when typedecls
		  (mapcar #'(lambda (d) (change-class d 'formal-type-decl))
		    (declarations typedecls)))
    'var-decls (when vardecls
		 (mapcar #'(lambda (d) (change-class d 'formal-const-decl))
		   (declarations vardecls)))))

(defun mk-sal-typedecls (attrs &rest typedecls)
  (make-sal-instance 'type-decls attrs
    'declarations typedecls))

(defun mk-sal-vardecls (attrs &rest vardecls)
  (make-sal-instance 'var-decls attrs
    'declarations vardecls))

(defun mk-sal-contextbody (attrs &rest decls)
  (make-sal-instance 'context-body attrs
    'declarations decls))

(defun mk-sal-constantdeclaration (attrs id formals type &optional definition)
  (make-sal-instance 'constant-declaration attrs
    'id (id id)
    'formals (when (declarations formals)
	       (mapcar #'(lambda (d)
				 (change-class d 'bind-decl))
		       (declarations formals)))
    'declared-type type
    'definition definition))

(defun mk-sal-typedeclaration (attrs identifier &optional typedef)
  (when (typep typedef '(or datatype scalar-type))
    (setf (id typedef) (id identifier)))
  (make-sal-instance 'type-declaration attrs
		     'id (id identifier)
		     'type-expr typedef))

(defun mk-sal-contextdeclaration (attrs identifier contextname)
  (make-sal-instance 'context-declaration attrs
		     'id (id identifier)
		     'context-name contextname))

(defun mk-sal-contextname (attrs identifier &optional actualparameters)
  (make-sal-instance 'context-name attrs
		     'id (id identifier)
		     'actuals (when actualparameters
				(mapcar #'mk-sal-actual
				  (append (actual-types actualparameters)
					  (actual-exprs actualparameters))))))

(defmethod id ((x symbol))
  x)

(defmethod mk-sal-actual ((ex expression))
  (make-instance 'actual
    :expr ex))

(defmethod mk-sal-actual ((ex type-expression))
  (make-instance 'actual
    :expr ex
    ;;:type-value ex
    ))
    

(defun mk-sal-actualparameters (attrs actualtypes actualexprs)
  (make-sal-instance 'actual-parameters attrs
		     'actual-types (actual-types actualtypes)
		     'actual-exprs (actual-exprs actualexprs)))

(defun mk-sal-actualtypes (attrs &rest actualtypes)
  (make-sal-instance 'actual-types attrs
		     'actual-types actualtypes))

(defun mk-sal-actualexprs (attrs &rest actualexprs)
  (make-sal-instance 'actual-exprs attrs
		     'actual-exprs actualexprs))

(defun mk-sal-moduledeclaration (attrs id vardecls module)
  (make-sal-instance 'module-declaration attrs
    'id (id id)
    'parameters (mapcar #'(lambda (d)
			    (change-class d 'module-parameter-decl))
		  (declarations vardecls))
    'module module))

(defun mk-sal-basemodule (attrs &rest decls)
  (make-sal-instance 'base-module attrs
    'declarations (mapcan #'(lambda (d)
			      (if (state-var-decls? d)
				  (var-decls d)
				  (list d)))
		    decls)))

(defun mk-sal-inputdecl (attrs &rest decls)
  (mapc #'(lambda (d) (change-class d 'input-decl)) decls)
  (setf (section-last? (car (last decls))) t)
  (make-sal-instance 'input-decls attrs
    'var-decls decls))

(defun mk-sal-outputdecl (attrs &rest decls)
  (mapc #'(lambda (d) (change-class d 'output-decl)) decls)
  (setf (section-last? (car (last decls))) t)
  (make-sal-instance 'output-decls attrs
    'var-decls decls))

(defun mk-sal-globaldecl (attrs &rest decls)
  (mapc #'(lambda (d) (change-class d 'global-decl)) decls)
  (setf (section-last? (car (last decls))) t)
  (make-sal-instance 'global-decls attrs
    'var-decls decls))

(defun mk-sal-localdecl (attrs &rest decls)
  (mapc #'(lambda (d) (change-class d 'local-decl)) decls)
  (setf (section-last? (car (last decls))) t)
  (make-sal-instance 'local-decls attrs
    'var-decls decls))

(defun mk-sal-defdecl (attrs &rest definitions)
  (make-sal-instance 'def-decl attrs
    'definitions definitions))

(defun mk-sal-initdecl (attrs &rest defs-or-commands)
  (make-sal-instance 'init-decl attrs
    'definitions-or-commands
    (mapcan #'(lambda (x)
		(typecase x
		  (simple-definition (list x))
		  (forall-definition (list x))
		  (t (setf (last-assignment? (car (last (commands x)))) t)
		     (commands x))))
      defs-or-commands)))

(defun mk-sal-transdecl (attrs &rest defs-or-commands)
  (make-sal-instance 'trans-decl attrs
    'definitions-or-commands
    (mapcan #'(lambda (x)
		(typecase x
		  (simple-definition (list x))
		  (forall-definition (list x))
		  (t (setf (last-assignment? (car (last (commands x)))) t)
		     (commands x))))
      defs-or-commands)))

(defun mk-sal-simpledefinition (attrs lhs rhs)
  (make-sal-instance 'simple-definition attrs
		     'lhs lhs
		     'rhs-definition rhs))

(defun mk-sal-arrayaccess ()
  (break))

(defun mk-sal-recordaccess ()
  (break))

(defun mk-sal-tupleaccess ()
  (break))

(defun mk-sal-rhsexpression (attrs expr)
  (declare (ignore attrs))
  expr)

(defun mk-sal-rhsselection (attrs expr)
  (make-sal-instance 'rhs-selection attrs
		     'expression expr))

(defun mk-sal-foralldefinition (attrs bindings &rest definitions)
  (make-sal-instance 'forall-definition attrs
		     'bindings (mapcar #'(lambda (b)
					   (change-class b 'bind-decl))
				 (if (var-decls? bindings)
				     (declarations bindings)
				     bindings))
		     'definitions definitions))

(defun mk-sal-labeledcommand (attrs label command)
  (make-sal-instance 'labeled-command attrs
		     'label label
		     'guarded-command command))

(defun mk-sal-label (attrs label)
  (declare (ignore attrs))
  label)

(defun mk-sal-guardedcommand (attrs guard assignments)
  (make-sal-instance 'guarded-command attrs
		     'guard guard
		     'assignments (definitions assignments)))

(defun mk-sal-guard (attr expr)
  (make-sal-instance 'guard attr
		     'expression expr))

(defun mk-sal-assignments (attrs &rest defs)
  (make-sal-instance 'assignments attrs
		     'definitions defs))

(defun mk-sal-somecommands (attrs &rest commands)
  (make-sal-instance 'some-commands attrs
		     'commands commands))

(defun mk-sal-multicommand (attrs vardecls command)
  (make-sal-instance 'multi-command attrs
		     'var-decls vardecls
		     'some-command command))

(defun mk-sal-synchronouscomposition (attrs mod1 mod2)
  (make-sal-instance 'synchronous-composition attrs
		     'module1 mod1
		     'module2 mod2))

(defun mk-sal-asynchronouscomposition (attrs mod1 mod2)
  (make-sal-instance 'asynchronous-composition attrs
		     'module1 mod1
		     'module2 mod2))

(defun mk-sal-multisynchronous (attrs vardecl module)
  (make-sal-instance 'multi-synchronous attrs
		     'var-decl vardecl
		     'module module))

(defun mk-sal-multiasynchronous (attrs vardecl module)
  (make-sal-instance 'multi-asynchronous attrs
		     'var-decl vardecl
		     'module module))

(defun mk-sal-hiding (attrs ids module)
  (make-sal-instance 'hiding attrs
		     'ids ids
		     'module module))

(defun mk-sal-newoutput (attrs vardecls module)
  (make-sal-instance 'new-output attrs
		     'var-decls vardecls
		     'module module))

(defun mk-sal-withmodule (attrs vardecls module)
  (make-sal-instance 'with-module attrs
		     'new-var-decls (mapcan #'(lambda (d)
						(if (state-var-decls? d)
						    (var-decls d)
						    (list d)))
				      vardecls)
		     'module module))

(defun mk-sal-newvardecls (attrs &rest decls)
  (declare (ignore attrs))
  decls)

(defun mk-sal-renaming (attrs renames module)
  (make-sal-instance 'renaming attrs
		     'renames renames
		     'module module))

(defun mk-sal-renames (attrs &rest renames)
  (declare (ignore attrs))
  renames)

(defun mk-sal-rename (attrs lhs rhs)
  (make-sal-instance 'rename attrs
		     'lhs lhs
		     'rhs rhs))

(defun mk-sal-moduleinstance (attrs modname actuals)
  (make-sal-instance 'module-instance attrs
		     'mod-name modname
		     'actuals (when actuals (actuals actuals))))

(defun mk-sal-moduleactuals (attrs &rest actuals)
  (make-sal-instance 'module-actuals attrs
		     'actuals actuals))

(defun mk-sal-modulename (attrs id)
  (assert (and id (symbolp id)))
  (make-sal-instance 'module-name attrs
		     'id id))

(defun mk-sal-qualifiedmodulename (attrs id context-name)
  (make-sal-instance 'qualified-module-name attrs
		     'id id
		     'context-name context-name))

(defun mk-sal-observemodule (attrs module observer)
  (make-sal-instance 'observe-module attrs
		     'module1 module
		     'module2 observer))

;; ASHISH: Copied from hybrid-sal.lisp
(defun mk-sal-invardecl (attrs expr)
  (make-sal-instance 'invar-decl attrs
    'expression expr))

(defun mk-sal-initfordecl (attrs expr)
  (make-sal-instance 'initfor-decl attrs
    'expression expr))
;; ASHISH: End of new stuff

(defun mk-sal-scalartype (attrs &rest scalarelements)
  (make-sal-instance 'scalar-type attrs
    'identifiers (mapcar #'id scalarelements)))

(defun mk-sal-scalarelement (attrs id)
  (make-sal-instance 'scalar-element attrs 'id id))

(defun mk-sal-datatype (attrs &rest constructors)
  (make-sal-instance 'datatype attrs
		     'constructors constructors))

(defun mk-sal-constructor (attrs identifier &rest accessors)
  (make-sal-instance 'constructor attrs
		     'id (id identifier)
		     'arguments accessors
		     'recognizer (intern (format nil "~a?" (id identifier)))))

(defun mk-sal-accessor (attrs identifier type)
  (make-sal-instance 'accessor attrs
		     'id (id identifier)
		     'declared-type type))

(defun mk-sal-typename (attrs identifier)
  (assert (and identifier (symbolp identifier)))
  (make-sal-instance 'type-name attrs
		     'id identifier))

(defun mk-sal-qualifiedtypename (attrs id context-name)
  (make-sal-instance 'qualified-type-name attrs
		     'id id
		     'context-name context-name))

(defun mk-sal-subrange (attrs low high)
  (make-sal-instance 'subrange attrs
		     'lower-bound low
		     'upper-bound high))

(defun mk-sal-unbounded ()
  (break))

(defun mk-sal-arraytype (attrs indextype type)
  (make-sal-instance 'array-type attrs
		     'domain indextype
		     'range type))

(defun mk-sal-tupletype (attrs &rest types)
  (make-sal-instance 'tuple-type attrs
		     'types types))

(defun mk-sal-recordtype (attrs &rest fields)
  (make-sal-instance 'record-type attrs
		     'fields fields))

(defun mk-sal-fielddeclaration (attrs id type)
  (make-sal-instance 'field-declaration attrs
		     'id id
		     'declared-type type))

(defun mk-sal-functiontype (attrs domain range)
  (make-sal-instance 'function-type attrs
		     'domain domain
		     'range range))

(defun mk-sal-statetype (attrs module)
  (make-sal-instance 'module-state-type attrs
		     'module module))

(defun mk-sal-modinit (attrs module)
  (make-sal-instance 'module-init attrs
		     'module module))

(defun mk-sal-modtrans (attrs module)
  (make-sal-instance 'module-trans attrs
		     'module module))

(defun mk-sal-nextoperator (attrs name)
  (make-sal-instance 'next-operator attrs
		     'name name))

(defun mk-sal-nameexpr (attrs id)
  (assert (symbolp id))
  (make-sal-instance 'name-expr attrs
		     'id id))

(defun mk-sal-qualifiednameexpr (attrs identifier context-name)
  (make-sal-instance 'qualified-name-expr attrs
		     'id (id identifier)
		     'context-name context-name))

(defun mk-sal-application (attrs operator argument)
  (let ((infix? (getf attrs 'infix?))
	(unary? (getf attrs 'unary?)))
    (remf attrs 'infix?)
    (remf attrs 'unary?)
    (make-sal-instance (sal-application-class operator infix? unary?)
	attrs
      'operator operator
      'argument argument)))
    ;(if (or (not (tuple-literal? argument))
			;(cdr (exprs argument)))
		    ;argument
		    ;(car (exprs argument)))

(defun sal-application-class (operator infix? unary?)
  (case (id operator)		;; ASHISH: MAJOR BUG FIXED HERE
    (NOT (if unary? 'unary-negation 'negation))
    (AND (if infix? 'infix-conjunction 'conjunction))
    (OR  (if infix? 'infix-disjunction 'disjunction))
    ;;(XOR (if infix? 'infix-xor-expr 'xor-expr))
    (=>  (if infix? 'infix-implication 'implication))
    (<=> (if infix? 'infix-iff 'iff))
    (=   (if infix? 'infix-equation 'equation))
    (/=  (if infix? 'infix-disequation 'disequation))
    (t   (cond (infix? 'infix-application)
	       (unary? 'unary-application)
	       (t 'application)))))

(defun mk-sal-arrayselection (attrs array index)
  (make-sal-instance 'array-selection attrs
    'operator array
    'argument index))

(defun mk-sal-recordselection (attrs record field)
  (make-sal-instance 'record-selection attrs
    'argument record
    'id (id field)))

(defun mk-sal-tupleselection (attrs tuple index)
  (make-sal-instance 'tuple-selection attrs
    'argument tuple
    'index (this-number index)))

(defun mk-sal-recordliteral (attrs &rest entries)
  (make-sal-instance 'record-literal attrs
		     'assignments entries))

(defun mk-sal-recordentry (attrs id expr)
  (make-sal-instance 'record-entry attrs
		     'id (id id)
		     'expression expr))

(defun mk-sal-tupleliteral (attrs &rest exprs)
  (make-sal-instance 'tuple-literal attrs
		     'exprs exprs))

(defun mk-sal-updateexpression (attrs expr updateposition value)
  (make-sal-instance 'update-expression attrs
		     'expression expr
		     'updateposition updateposition
		     'value value))

(defun mk-sal-arrayposition ()
  (break))

(defun mk-sal-arrayliteral (attrs binding expr)
  (make-sal-instance 'array-literal attrs
		     'bindings (list binding)
		     'expression expr))

(defun mk-sal-indexvardecl (attrs id type)
  (make-sal-instance 'index-var-decl attrs
		     'id (id id)
		     'declared-type type))

(defun mk-sal-lambdaabstraction (attrs bindings expr)
  (make-sal-instance 'lambda-abstraction attrs
    'bindings (mapcar #'(lambda (b)
			  (change-class b 'bind-decl))
		(if (var-decls? bindings)
		    (declarations bindings)
		    bindings))
    'expression expr))

(defun mk-sal-quantifiedexpression (attrs quant bindings expr)
  (make-sal-instance 'quantified-expression attrs
		     'quantifier quant
		     'bindings (mapcar #'(lambda (b)
					   (change-class b 'bind-decl))
				 (if (var-decls? bindings)
				     (declarations bindings)
				     bindings))
		     'expression expr))

(defun mk-sal-quantifier (attrs quant)
  (declare (ignore attrs))
  quant)

(defun mk-sal-letexpression (attrs declarations expression)
  (make-sal-instance 'let-expression attrs
		     'declarations declarations
		     'expr expression))

(defun mk-sal-letdeclarations (attrs &rest let-decls)
  (declare (ignore attrs))
  let-decls)

(defun mk-sal-letdeclaration (attrs id type value)
  (make-sal-instance 'let-declaration attrs
		     'id (id id)
		     'declared-type type
		     'value value))

(defun mk-sal-setpredexpression (attrs id type pred)
  (make-sal-instance 'set-pred-expr attrs
		     'bindings (list (make-sal-instance 'var-decl nil
							'id (id id)
							'declared-type type))
		     'expression pred))

(defun mk-sal-setlistexpression (attrs &rest exprs)
  (make-sal-instance 'set-list-expr attrs
		     'expressions exprs))

(defun mk-sal-conditional (attrs cond then else)
  (let ((elsif? (getf attrs 'elsif?)))
    (remf attrs 'elsif?)
    (make-sal-instance (if elsif?
			   'chained-conditional
			   'conditional)
	attrs
      'operator (make-sal-instance 'name-expr nil 'id 'IF)
      'argument (make-sal-instance 'tuple-literal nil
		  'exprs (list cond then else)))))

(defun mk-sal-numeral (attrs num)
  (make-sal-instance 'numeral attrs
		     'this-number (parse-integer (string num))))

(defun mk-sal-vardecl (attrs id type)
  (make-sal-instance 'var-decl attrs
		     'id (id id)
		     'declared-type type))

(defun mk-sal-typedecl (attrs identifier)
  (make-sal-instance 'formal-type-decl attrs
		     'id (id identifier)))

(defun mk-sal-identifiers (attrs &rest ids)
  (declare (ignore attrs))
  ids)

(defun mk-sal-identifier (attrs id)
  (apply #'make-instance
    'identifier
    'id id
    attrs))

(defun mk-sal-assertiondeclaration (attrs id form assertion)
  (make-sal-instance 'assertion-declaration attrs
		     'id (id id)
		     'assertion-form (if (symbolp form)
					 form
					 (form form))
		     'assertion assertion))

(defun mk-sal-assertionform (attrs form)
  (make-sal-instance 'assertion-form attrs
		     'form form))

(defun mk-sal-assertionproposition (attrs operator &rest assertions)
  (make-sal-instance 'assertion-proposition attrs
		     'operator operator
		     'assertions assertions))

(defun mk-sal-assertionoperator (attrs operator)
  (declare (ignore attrs))
  operator)

(defun mk-sal-quantifiedassertion (attrs quantifier vardecls assertion)
  (make-sal-instance 'quantified-assertion attrs
		     'quantifier quantifier
		     'var-decls vardecls
		     'assertion assertion))

(defun mk-sal-modulemodels (attrs module assertion)
  (make-sal-instance 'module-models attrs
		     'module module
		     'assertion assertion))

(defun mk-sal-moduleimplements (attrs module1 module2)
  (make-sal-instance 'module-implements attrs
		     'module1 module1
		     'module2 module2))

;; -----------------------------------------------------

