;; --------------------------------------------------------------------
;; HybridSAL
;; Copyright (C) 2006, SRI International.  All Rights Reserved.
;; 
;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License
;; as published by the Free Software Foundation; either version 2
;; of the License, or (at your option) any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
;; --------------------------------------------------------------------

;;;;;;;;;;;;;;;;;;;;;;;;;;;;; -*- Mode: Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; sal-extension.lisp --
;; Author          : Ashish Tiwari
;; HISTORY
;; 05.29.07: Things required by abstractor....
;; 03.05.09: bool-atomic-fmla? All bool name-expr are bool-atomic-fmla.
;; 11.09.10: added (defmethod get-stringR* ((rexpr tuple-literal))...)
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package :sal)

;(export '(ps-eq op2symbol args1 args2 binary? pvsexpr2RAFstrings))

(defconstant *times-op* (make-sal-instance 'name-expr nil 'id '*))
(defconstant *divides-op* (make-sal-instance 'name-expr nil 'id '/))
(defconstant *not-op* (make-sal-instance 'name-expr nil 'id 'NOT))


;; ============================================================================
;; Return the listOfPolynomials in atomic formulas in the given property
;; Eg. G( x > 5) returns the list '("x - 5")
;; ============================================================================
(defun pvsexpr2RAFstrings (expr)
  (get-predicates1S2B* expr))

(defmethod get-predicates1S2B* ((expr application))
  (let ((ope (operator expr)))
    (if (or (eq (id ope) 'G) (eq (id ope) 'F))
	(get-predicates1B* (car (exprs (argument expr))))
	(sal-error t "Unidentified LTL operator~A~%" ope))))

(defmethod get-predicates1B* ((expr application))
  (case (id (operator expr))
	((AND OR => <=>) 
	 (nconc (get-predicates1B* (args1 expr))
		(get-predicates1B* (args2 expr))))
	((NOT G F) (get-predicates1B* (args1 expr)))
	((< <= >= > =)
	 (if (tc-eq (args1 expr) 'BOOLEAN) nil
	 (let ((str1 (get-stringR* (args1 expr)))
	       (str2 (get-stringR* (args2 expr))))
	   (list (format nil "~A - (~A)" str1 str2)))))
	(t (sal-error t "PANIC: What the hell is this expr ~A~%" expr))))

(defmethod get-predicates1B* ((expr t))
  (format t "Error: Missing code. get-predicates1B* on expr~%")
  (break))

(defmethod get-stringR* ((rexpr infix-application))	;; SAL?
  (case (id (operator rexpr))
	((+ - * /)
	 (format nil "(~A) ~A (~A)"
		(get-stringR* (args1 rexpr))
		(operator rexpr)
		(get-stringR* (args2 rexpr))))
	(t (sal-error t "PANIC: What the hell is this expr ~A~%" rexpr))))

(defmethod get-stringR* ((rexpr application))		;; SAL??
  (case (id (operator rexpr))
	(- (format nil "-~A" (get-stringR* (argument rexpr))))
	(t (sal-error t "PANIC: What the hell is this expr ~A~%" rexpr))))

;; -(3) parses as (unary-application (argument (tuple-literal (exprs.....))))
(defmethod get-stringR* ((rexpr tuple-literal))
  (let ((args (sal:exprs rexpr)))
    (when (cdr args)
  	(describe rexpr)
	(sal-error t "PANIC: Expecting 1 argument, found multiple args:~A~%" rexpr)
  	(break))
    (get-stringR* (car args))))
  
(defmethod get-stringR* ((real-expr numeral))		;; numerals
  (format nil "~a" (this-number real-expr)))

(defmethod get-stringR* ((real-expr name-expr))		;; numerals, name-exprs
  (format nil "~a" (id real-expr)))

(defmethod get-stringR* ((expr t))		;; numerals, name-exprs
  (describe expr)
  (sal-error t "ERROR: Missing code perhaps. Unknown expr~%")
  (break))
;; ============================================================================

;; ============================================================================
;; Functions on Real Atomic Formulas.
;; ============================================================================
(defun real-atomic-fmla? (expr)
  (and (tc-eq expr 'BOOLEAN)
  (cond ((infix-application? expr)		;; OR sal:infix-appli??
	 (tc-eq (args1 expr) 'REAL))
	((unary-negation? expr)		;; NEW for negation!!!
	 (real-atomic-fmla? (args1 expr))))))

(defun bool-atomic-fmla? (expr)
  (and (tc-eq expr 'BOOLEAN)
  (cond ((infix-equation? expr)		;; x = TRUE
	 (tc-eq (args1 expr) 'BOOLEAN))
	((unary-negation? expr)		;; NEW for negation!!!
	 (bool-atomic-fmla? (args1 expr)))
	((name-expr? expr)
	 ;(or (eq (id expr) 'TRUE) (eq (id expr) 'FALSE))
	 t))))
;; ============================================================================

;; =========================================================
(defun file-exists-p (sal-file)
  (probe-file sal-file))

(defun times-operator () *times-op*)
(defun divides-operator () *divides-op*)

(defmethod conjuncts ((fmla unary-negation))
  (mapcar #'negate  (disjuncts (args1 fmla))))
(defmethod conjuncts ((fmla infix-conjunction))
  (nconc (conjuncts (args1 fmla)) (conjuncts (args2 fmla))))
(defmethod conjuncts ((fmla t))
  (list fmla))

(defmethod disjuncts ((fmla unary-negation))
  (mapcar #'negate (conjuncts (args1 fmla))))
(defmethod disjuncts ((fmla infix-disjunction))
  (nconc (disjuncts (args1 fmla))(disjuncts (args2 fmla))))
(defmethod disjuncts ((fmla infix-implication))
  (nconc (mapcar #'negate (conjuncts (args1 fmla)))
         (disjuncts (args2 fmla))))
(defmethod disjuncts ((fmla t))
  (list fmla))

(defun negate (fmla)
  (let ((attrs (list 'unary? t)))
    (mk-sal-application attrs *not-op* 
      (mk-sal-tupleliteral nil fmla))))
;; =========================================================

;; =========================================================
(defmethod baf2boolvar ((expr unary-negation))
  (baf2boolvar (args1 expr)))
(defmethod baf2boolvar ((expr infix-equation)) 	;; var = val assumed
  (baf2boolvar (args1 expr)))
(defmethod baf2boolvar ((expr name-expr)) 	;; var = val assumed
  (if (or (eq (id expr) 'TRUE)
	  (eq (id expr) sal:'FALSE)) 
      'TRUE
      (id expr)))
(defmethod baf2boolvar ((expr t))
  (sal-error t "Unknown sal bool atomic fmla~%"))
  
(defmethod baf2boolval ((expr infix-equation))
  (baf2boolval (args2 expr)))
(defmethod baf2boolval ((expr unary-negation))
  (baf2boolval (args1 expr)))
(defmethod baf2boolval ((expr name-expr)) ;; TRUE/FALSE/var
  (cond ((eq (id expr) 'TRUE) 'TRUE)
        ((eq (id expr) 'FALSE) 'FALSE)
	;(t (id expr))
	(t 'TRUE)))
(defmethod baf2boolval ((expr t))
  (sal-error t "Unknown sal bool atomic fmla~%"))

(defmethod baf2boolsgn ((expr unary-negation))
  (not (baf2boolsgn (args1 expr))))	;; equation
(defmethod baf2boolsgn ((expr t))
  t)
;; =========================================================
