;; --------------------------------------------------------------------
;; HybridSAL
;; Copyright (C) 2006, SRI International.  All Rights Reserved.
;; 
;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License
;; as published by the Free Software Foundation; either version 2
;; of the License, or (at your option) any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
;; --------------------------------------------------------------------

;;;;;;;;;;;;;;;;;;;;;;;;;;* -*- Mode: Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; vim: syntax=lisp
;; hsal-compose.lisp --
;; Author          : Ashish Tiwari
;; Created On      : Thu May 22, 2002
;; Last Modified By: Ashish Tiwari
;; Last Modified On: Thu May 22, 2002
;; Update Count    : 0
;; Status          : Unknown, use with caution
;;
;; HISTORY : 
;; Generate all feasibility information statically.
;; 06.10.03: remove-big-small added.
;; 06.10.03: local-infeas0: return value -1 indicates *f1*/*f2* was reset.
;; 06.10.03: remove-big-small* :: apply as DEFAULT????????
;; 08.29.03: remove-small: if len < 4, do nothing. x=0 allowed to go in.
;; 09.02.03: global-feas? introduced.
;; 06.05.07: WE WILL NEVER RESET *f1*...else known? crashes
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(provide 'decision-procedure-feas)		;; This module is named decision-procedure
(cl:defpackage "decision-procedure-feas"
  (:nicknames "dp-feas")
  (:export "get-feasible-states" "known?" "global-feas?"
	   "FEAS-ff" "FEAS-tt" "FEAS-pols"	;; Structure returned.
	   "set-feas-debug-level")
  (:use "polynomial-representation-core" "chaining-dp" "cl" "user" "clos"))

(in-package "decision-procedure-feas")	;; The "feasibility" extension of dp-core

;; ---------------------------------------------------------------------------
;; get-feasible-states: Describe all feasible states as a SAL fn.
;; ---------------------------------------------------------------------------
;; This is the interface with the rest of the abstractor code.

(defstruct FEAS ff tt pols)
(defvar *dlevel* 4)		;; Default nil. Set using accessors

;; Algo: grab one poly, grab all with the SAME vars, generate all feas.
;; Until all pols exhausted. RETURN: FEAS-List
(defun get-feasible-states (pols &optional (namap nil) (E0 nil) (R0 nil) (S0 nil))
  (let ((*negs* (if namap namap (loop for i in pols collect (cons i (prep:polyrepNegativePoly i))))))
    (declare (special *negs*))
    (let ((feasLL (local-infeas pols E0 R0 S0)))
      (print-feas feasLL)
      (apply #'nconc feasLL))))

;; ----------------------------------------------------------------------------
;; Debug print-debug 2 option.
;; ----------------------------------------------------------------------------
(defun set-feas-debug-level (n)
  (setf *dlevel* n))

(defmacro print-debug (level &rest others)
  `(if (and *dlevel* (> ,level *dlevel*))
       (funcall #'format ,@others)))
;; ----------------------------------------------------------------------------

;; ---------------------------------------------------------------------------
;; local-infeas*: detect all local infeasibilities.
;; ---------------------------------------------------------------------------
(defun local-infeas (pols E R S)
  (let* ((feasLPolsVarsL (local-infeas2 pols E R S))
	 (feasLL2 (cross-infeas2 feasLPolsVarsL E R S))
	 (feasLL1 (mapcar #'car feasLPolsVarsL)))
    (nconc feasLL1 feasLL2)))

(defun local-infeas2 (pols E R S)
  (local-infeas2* pols nil E R S))

;; unprocessed pols, processed partitions, infeasibilities, feasibilities
;; RETURN: (FEAS-list, Pols, vars)-list
(defun local-infeas2* (unpols res E R S)
  (if (null unpols) (return-from local-infeas2* res))
  (let* ((pol1 (car unpols))
	 (var1 (prep:allVarsIn pol1))
	 (pols (prep:getExactlyRelevantPols unpols var1))
	 (feas (local-infeas1* pols nil E R S)))
    (local-infeas2* (set-difference unpols pols) (cons (list feas pols var1) res) E R S)))

;; divide into groups of 7 and test infeasibility: Overlap 3!! RETURN: FEAS-list
(defun local-infeas1* (unpols res E R S)
  (if (null unpols) (return-from local-infeas1* res))
  (multiple-value-bind (pols proc-pols) 
	(if (> (length unpols) 7) (values (subseq unpols 0 7) (subseq unpols 0 4)) (values unpols unpols))
    (local-infeas1* (set-difference unpols proc-pols) (cons (local-infeas0 pols E R S) res) E R S)))

;; Return value: list-of-infeas-combinations, list-of-feas-states; all over pols!
(defun local-infeas0 (pols E R S &key (db nil))
  (let ((*f1* t) (*f2* t) (*db* db) (*all* (append E R S))) ;; (*n* (expt 3 (length pols)))
    (declare (special *f1* *f2* *db* *all*))	;; *db* : previous results to use!!
    (print-debug 5 t " Searching space of size ~A...~%" (expt 3 (length pols)))
    (print-debug 5 t "  ~A~%" (mapcar #'prep:polyrepPrint pols))
    (multiple-value-bind (infeas feas)
    	(local-infeas0* pols nil nil nil nil E R S)
      (print-debug 5 t " Searching space of size ~A...DONE.~%" (expt 3 (length pols)))
      (if (and *f2* (null feas)) (break))
      (make-FEAS :pols (reverse pols) :ff (if *f1* infeas -1) :tt (if *f2* feas -1)))))

;; pols: unprocessed pols. polstk: processed pol stack. curr: processed >,<,= stack.
;; infeas, feas: return values. *flag1,2*: infeas, feas maintained?
(defun local-infeas0* (pols polstk curr infeas feas E0 R0 S0)
  (declare (special *f1* *f2* *all* *negs*))
  (if (null pols) (return-from local-infeas0* (values infeas (if *f2* (cons curr feas)))))
  ; (if (and *f1* *f2* (> (length infeas) 100)) (progn (setf *f1* nil) (setf infeas nil) (print-debug 5 t "resetting f1~%")))
  (if (and *f1* *f2* (> (length feas) 100)) (progn (setf *f2* nil) (setf feas nil) (print-debug 5 t "resetting f2~%")))
  (let* ((p (car pols)) (negp (cdr (assoc p *negs*))))
    (if (null negp) (progn (print-debug 5 t "Negative Poly not found~%") (break)))
    (multiple-value-bind (st E1 R1 S1)
	 (feas? negp chaining-dp::'> polstk curr infeas E0 R0 S0)
      (if (some #'(lambda(x)(null (chaining-dp::POL-op x))) E1) (break))
      (multiple-value-bind (infeas1 feas1) 
		(if st 
		    (local-infeas0* (cdr pols) (cons p polstk) (cons chaining-dp::'< curr) infeas feas E1 R1 S1)
		    (values (if (and *f1* E1) (cons (set-difference E1 *all*) infeas) infeas) feas))
        (multiple-value-bind (st E2 R2 S2)
	 	(feas? p chaining-dp::'> polstk curr infeas1 E0 R0 S0)
	  (if (some #'(lambda(x)(null (chaining-dp::POL-op x))) E2) (break))
    	  (multiple-value-bind (infeas2 feas2) 
		(if st 
		    (local-infeas0* (cdr pols) (cons p polstk) (cons chaining-dp::'> curr) infeas1 feas1 E2 R2 S2)
		    (values (if (and *f1* E2) (cons (set-difference E2 *all*) infeas1) infeas1) feas1))
  	    (multiple-value-bind (st E3 R3 S3)
	 		(feas? p chaining-dp::'= polstk curr infeas2 E0 R0 S0)
	      (if (some #'(lambda(x)(null (chaining-dp::POL-op x))) E3) (break))
	      (if st
		  (local-infeas0* (cdr pols) (cons p polstk) (cons chaining-dp::'= curr) infeas2 feas2 E3 R3 S3)
		  (values (if (and *f1* E3) (cons (set-difference E3 *all*) infeas2) infeas2) feas2)))))))))

(defun feas? (pol op polstk curr infeas E R S)
  (declare (special *negs* *db*))
  (let* ((pols (cons pol polstk)) 
	 (ops (cons op curr))
	 (newpols (loop for i in pols as j in ops collect 
		    (if (eq j chaining-dp::'<) 
			(cons (let ((ans (cdr (assoc i *negs*))))
				(if ans ans (break))) chaining-dp::'>)
			(cons i j)))))
    (if (known? newpols infeas) (return-from feas? nil))
    (if (some #'(lambda(x) (known? newpols (FEAS-ff x))) *db*) (return-from feas? nil)))
  (chaining-dp:set-debug-level *dlevel*)
  (chaining-dp:set-optimize-flag t)
  (multiple-value-bind (st E1 R1 S1)
    (chaining-dp:saturate nil nil nil (list pol op) E R S)
    (when *dlevel*
      (print-debug 2 t "st = ~A~%E = ~%" st)
      (loop for i in E1 do (print-debug 2 t "~A~%" (chaining-dp::POL-pol i)))
      (print-debug 2 t "R =~%")
      (loop for i in R1 do (print-debug 2 t "~A~%" (chaining-dp::POL-pol i)))
      (print-debug 2 t "S =~%")
      (loop for i in S1 do (print-debug 2 t "~A~%" (chaining-dp::POL-pol i))))
    ;(if (and (eq (length pol) 7) (equal op chaining-dp::'>) (equal curr (list chaining-dp::'> chaining-dp::'= chaining-dp::'> chaining-dp::'>)))
	;(break))
    (values st E1 R1 S1)))

(defun global-feas? (poloplist feasL npol)
  (cond ((null npol) t)		;; new pol on top of poloplist is NOT a amap pol
	((eq npol -1) (global-feas?* poloplist feasL nil))	;; top(poloplist) is AMAP pol
	(t (global-feas?* poloplist feasL npol))))
      
(defun global-feas?* (poloplist feasL npol)
  (if (null feasL) t
      (let ((feas-ff (FEAS-ff (car feasL)))
	    (feas-pols (FEAS-pols (car feasL))))
	(if (and (not (member (caar poloplist) feas-pols))
		 (if npol (not (member npol feas-pols)) t))
	    (global-feas?* poloplist (cdr feasL) npol)
	    (if (not (eq feas-ff -1))
	    	(if (known? poloplist feas-ff) nil	;; known to be infeasible
		    (global-feas?* poloplist (cdr feasL) npol))
	    	(global-feas?* poloplist (cdr feasL) npol))))))
;; ---------------------------------------------------------------------------
  
;; ---------------------------------------------------------------------------
;; cross-infeas2: (feasL, Pols, Vars)-list E R S --> feasListList
;; Note: P1, P2...\in Pols s.t. V1,V2,...\in Vars. Pi over Vi.
;; ---------------------------------------------------------------------------
(defun cross-infeas2 (feasLPolsVarsL E R S)
  (print-debug 5 t " Checking for cross infeasibilities~%")
  (let ((parts1 (mapcar #'cadr feasLPolsVarsL))
	(vars (mapcar #'caddr feasLPolsVarsL)))
    (cross-infeas2* parts1 vars 0 (length vars) nil E R S)))

;; pols: equivalence class of pols. vars: corresponding variables. Return Val: FEAS-list-list
;; Algo: for each eq.class P1 in pols, get eq.classes P2,P3.. s.t.
;; V2,V3,.. are subsets of V1 *AND* {V2,V3,...} IS MINIMAL. Then, do 
;; "feas(p1,p2,...)" where pi\in Pi.
(defun cross-infeas2* (pols vars n m feasList E R S)
  (if (eq n m) (return-from cross-infeas2* feasList))
  (let* ((var1 (nth n vars))
	 (usepols (loop for i in vars as j in pols 
			if (and (subsetp i var1) (not (subsetp var1 i))) collect (cons j i)))
	 (relpols (delete-if-not #'(lambda(x) (minimal? x usepols)) usepols))
	 (finpols (mapcar #'car relpols))
	 (newFeasList (cross-infeas1-wrap* (nth n pols) finpols E R S)))
    (cross-infeas2* pols vars (+ n 1) m (cons newFeasList feasList) E R S)))

;; pols: eq.class P1; relpols: list-of eq.classes Pi s.t. Vi \subset V1, Vi minimal
(defun cross-infeas1-wrap* (pols relpols E R S)
  (if (or (null relpols) (null pols)) (return-from cross-infeas1-wrap* nil))
  (print-debug 5 t " Checking for cross infeasibilities: ~A cases...~%" (apply #'* (length pols) (mapcar #'length relpols)))
  (let* ((newrelpols (remove-big-small relpols))
	 (numcases (apply #'* (length pols) (mapcar #'length newrelpols))))
    (print-debug 5 t " After optimization, we are left with ~A cases...~%" numcases)
    (cross-infeas1 pols (if (> numcases 32) (remove-arbitrary newrelpols 16) newrelpols) E R S)))

;; Algo: get infeas info on all combinations (p1,p2,p3,..)
;; s.t. p1\in pols, pi\in (nth i-1 relpols)  AND THEN
;; get infeas info on (pols, p2, p3, ...) where p2,p3 as above.
(defun cross-infeas1 (pols relpols E R S)
  (let ((feasList (cross-infeas1* pols relpols E R S)))
    (if (> (length pols) 1)
        (cross-infeas1** pols relpols E R S feasList)
	feasList)))

;; pols: list of pols=P1. relpols: list of list-of pols={P2,P3,...}. Return Val: FEAS-list
(defun cross-infeas1* (pols relpols E R S &optional (feasList nil))
  (if (or (null relpols) (null pols)) (return-from cross-infeas1* feasList))
  (let ((newFeasList (cross-infeas1** (list (car pols)) relpols E R S feasList)))
    (cross-infeas1* (cdr pols) relpols E R S newFeasList)))

;; Return value: FEAS-list
;; Get infeas info on pols U {p1,p2,...} where
;; pi \in (nth i relpols)
(defun cross-infeas1** (pols relpols E R S res)
  (cond ((null relpols)
	 (cons (local-infeas0 pols E R S :db res) res))
	((null (car relpols))
	 res)
	(t
	 (let ((newres (cross-infeas1** (cons (caar relpols) pols) (cdr relpols) E R S res)))
	   (cross-infeas1** pols (cons (cdar relpols) (cdr relpols)) E R S newres)))))
;; ---------------------------------------------------------------------------

;; ---------------------------------------------------------------------------
;; Generic Functions?
;; ---------------------------------------------------------------------------
(defun known? (pols infeas)
  (some #'(lambda(x) (known?* pols x)) infeas))

(defun known?* (pols infeas)
  (if (< (length pols) (length infeas)) (return-from known?* nil))
    (known?** pols infeas))

(defun known?** (pols infeas)
  (if (null infeas) (return-from known?** t))
  (let* ((polPOL (car infeas))
	 (pol (chaining-dp::POL-pol polPOL))
	 (sgn (chaining-dp::POL-op polPOL))
	 (polsgn (member pol pols :test #'eq :key #'car)))
    (if (and polsgn (eq sgn (cdr (car polsgn))))
	(known?** pols (cdr infeas)) nil)))

(defun minimal? (poldotvar polsdotvars)
  (let ((this (cdr poldotvar)))
    (every #'(lambda(x)(or (eq (cdr x) this) (not (subsetp (cdr x) this)))) polsdotvars)))

(defun print-feas (feasLL)
  (print-debug 5 t " Feasibility Information:~%")
  (loop for i in feasLL do
  (loop for j in i do
    (print-debug 5 t " Polynomials:~%")
    (loop for l in (FEAS-pols j) do (print-debug 5 t "  ~A~%" (prep:polyrepPrint l)))
    (if (or (eq (FEAS-tt j) -1) (and (not (eq (FEAS-ff j) -1)) (< (length (FEAS-ff j)) (length (FEAS-tt j)))))
	(print-debug 5 t "  Length of FEAS-ff is ~A~%" (length (FEAS-ff j)))
	(print-debug 5 t "  Length of FEAS-tt is ~A~%" (length (FEAS-tt j))))
    ;(if (or (eq (FEAS-tt j) -1) (and (not (eq (FEAS-ff j) -1)) (< (length (FEAS-ff j)) (length (FEAS-tt j)))))
	;(loop for m in (FEAS-ff j) do 
	      ;(loop for l in m do (format t " ~A ~A 0, " (chaining-dp::POL-pol l) (chaining-dp::POL-op l)))
	      ;(format t "~%"))
  	;(loop for l in (FEAS-tt j) do (format t "~A~%" l)))
	)))

(defun remove-arbitrary (polLL num)
  (let ((numcases (apply #'* (mapcar #'length polLL))))
    (if (> numcases num) (remove-arbitrary (append (cdr polLL) (list (cdar polLL))) num) polLL)))
  ;(multiple-value-bind (old new) (loop for i in polLL as j = (remove-big-small* i)
	;if (and (> (length i) 1) j) return (values i j))
    ;(if old (cons new (remove old polLL :count 1))
	;(append (cdr polLL) (list (cdar polLL)))))

(defun remove-big-small (polLL)
  (mapcar #'(lambda(x) (remove-small* (remove-big* x))) polLL))

(defun remove-big* (pols)
  (let ((ans (remove-if #'(lambda(x) (> (length x) 2)) pols)))  ;; :count 1 previously
    (if ans ans 
        (let ((ans (remove-if #'(lambda(x) (> (length x) 2)) pols :count 1)))
    	  (if ans ans pols)))))

(defun remove-small* (pols)
  (if (< (length pols) 4) (return-from remove-small* pols))
  (let ((ans (remove-if #'(lambda(x) (eq (length x) 1)) pols))) ;; :count 1 previously
    (if ans ans 
        (let ((ans (remove-if #'(lambda(x) (eq (length x) 1)) pols :count 1)))
    	  (if ans ans pols)))))
;(defun remove-big-small-obsolete* (pols)
  ;(let ((big? #'(lambda(x) (> (length x) 2))))
    ;(if (some big? pols)
	;(remove-if big? pols :count 1)
  	;(let ((small? #'(lambda(x) (eq (length x) 1))))
  	  ;(if (some small? pols) 
	      ;(remove-if small? pols :count 1)
	      ;nil)))))
;; ---------------------------------------------------------------------------
