;; --------------------------------------------------------------------
;; HybridSAL
;; Copyright (C) 2006, SRI International.  All Rights Reserved.
;; 
;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License
;; as published by the Free Software Foundation; either version 2
;; of the License, or (at your option) any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
;; --------------------------------------------------------------------

;;;;;;;;;;;;;;;;;;;;;;;;;;* -*- mode: lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; vim: syntax=lisp
;; decide.lisp --
;; Author          : Ashish Tiwari
;; Created On      : Wed Apr 24, 2002
;; Last Modified By: Ashish Tiwari
;; Last Modified On: Wed Apr 24, 2002
;; Update Count    : 0
;; Status          : Unknown, use with caution
;;
;; HISTORY :
;; 04.27.02: Add check for redundancy!!! We are deducing same things!!
;; Also TODO : general unification in chaining sh'd be used; the non-
;; negativity of the unifier should be checked against the redundancy check.
;; OR, compute unifier, and then see what "can" be used for that unifier!!!
;; 04.30.02: The algo for chaining: 1. compute overlap mu0, nu0. 2. Check
;; for an inequality c0mu0 > p0 and d0nu0 > q0 in R s.t. c0,d0 > 0.
;; Multiply first by c0mu0 - p0 and get CP. For termination, right now,
;; we only shall compute chaining inferences when one of mu0, nu0 = I.
;; IDEA: Can prove termination of chaining inference in GENERAL??!!!!! No!
;; 04.30.02: paramtypes will be asserted as regular facts.
;; 04.30.02: Bug1--- *R* and *E* not updated.
;; Bug2: simplifyRbyE --- should be made incremental
;; Bug3: Simplification and Deletion rules are missing
;; 04.30.02: Dynamic variants added----Read code once!!
;; 05.07.02: Added consistencyChecks in decideNonLinearFourierMotzkin.
;; 05.08.02: Need to add REDUNDANT checks -- indispensable....
;; 05.09.02: Added dynamicSimplifyRbyErec in dynamicLinear... etc.
;; 05.15.02: Bug corrected in findRule! Search poly with Leading-Coef>0!
;; 05.16.02: chaining* (superposition) made strict.
;; 05.16.02: *optimize* added in findRule (returns on binomials now)
;; 05.16.02: leading coeff=1.
;; 05.20.02: redundant made more general to ensure termination!
;; 05.23.02: bug corrected in chainingPartial* (if ans => if factor)
;; TODO: IN (defun decideGeneralDynamicR (rule E R)) see commented line!
;; TODO: Change from LPO to Total-degree LPO
;; 06.06.02: decideNonLinearInequalitiesModE: swapped cond cases.
;; 06.06.02: findRule: extra "sgn" argument.
;; 06.07.02: decideNonLinearFM: If *optimize* then deduction->simplification.
;; 06.11.02: Cool idea: Inferences which result in EQUAL leading PP deleted!!
;; 	Deletion of inequalities has been temporarily revoked.
;; 06.11.02: changing return value of redundant: (yes, newE, newR)
;; 06.11.02: dynamicSimplify: leading coeff made 1/-1.
;; 06.11.02: decideLinearDynamicE: minor change, see comment there.
;; 06.12.02: What to return on EARLY TERMINATION? Passing defaults!
;; 06.18.02: Moving linearPoly? functions from polyrep-misc to here.
;; 06.20.02: findRuleParity: checks for second match before returning.
;; 06.20.02: *flag* special variable triggers strong optimizations.
;;	     See flag in decideNonLinearInequalitiesModE functions as well.
;; 07.09.02: Changing findRule to findRule2.
;; 07.09.02: Added a special variable *mu*: largestPP on which chained.
;; 10.17.02: Removed optimization from decideNonLinearInequalitiesModE...
;;	     they don't make sense to me anyway. Need to remove for V2V too!
;; 11.20.02: S0 added. Saturate return value BUG corrected.
;; 11.22.02: consistencyCheck optimized!
;; 11.24.02: chainPartialAllWith1 added: because R OR S rules can be chosen
;;		in findRule2 sometimes. NOTE: deduction rules--more incomplete?
;; 05.13.03: Bug on line 819 corrected. Recursive call to chainPartialAllWith1!!
;; 05.15.03: Bug in splitOnVar: flag added
;; 06.11.03: decideNonLinearInequalitiesModE: HARD LIMIT for termination enforced.
;; 06.14.03: print-debug introduced. BUG in optimizedChoice? corrected.
;; 07.09.03: Introduced interval?
;; 07.09.03: IMPORTANT: if new-fmla is linear, turn off OPTIMIZE flag.
;; 07.10.03: New nnormalizePoly (Destructive), not used.
;; 07.10.03: decideGeneralDynamicE: EInv: witness corrected.
;; 07.15.03: *bound-cache* introduced. Bug in getUpperBound INFERENCE RULE!!
;; 07.15.03: Trying to do some factorization!
;; 07.16.03: getLUBoundVar: witness was not a LIST. Return value corrected.
;; 07.17.03: Bug in redundant*: default case, return value (nil R)!!
;; 07.17.03: decideNonLinearInequalitiesModE1: flag=(firstIteration?) removed!!
;; 07.22.03: destructive dynamicGrobner changed to nondestructive **important**
;; 08.25.03: reset-wit added. Used by hsal-abs-real.lisp
;; 11.11.10: :exported decideDynamicS
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(provide 'decision-procedure)		;; This module is named decision-procedure
(cl:defpackage "decision-procedure-core"
  (:nicknames "chaining-dp")
  (:export "decideDynamicE" "decideDynamicR" "decideDynamicS" "check-invariant-new"
	   "satisfiable" "inconsistent"	;; Symbols returned by decideDynamic.
	   "saturate" "set-optimize-flag" "set-debug-level" "reset-wit")
  (:use "polynomial-representation-core" "cl" "user" "clos"))

(in-package "decision-procedure-core")	;; The core procedure

(defstruct POL pol wit op)	;; polynomial and witness

;; REQUIRES: prep:get-order
;; Set prep:set-* and chaining-dp:set-optimize before using!!!

(defvar *optimize* t)		;; Default t. Set and Get using accessors
(defvar *dlevel* nil)		;; Default t. Set and Get using accessors

;; ----------------------------------------------------------------------------
;; Debug print-debug 2 option.
;; ----------------------------------------------------------------------------
(defmacro print-debug (level &rest others)
  `(if (and *dlevel* (> ,level *dlevel*))
       (funcall #'format ,@others)))
;; ----------------------------------------------------------------------------

;; ----------------------------------------------------------------------------
;; Interface Functions.
;; Given a list of polynomials with operators >, <, or =,
;; decide if they are simultaneously satisfiable.
;; ----------------------------------------------------------------------------
(defun saturate (E R &optional (S nil) (ER nil) (E0 nil) (R0 nil) (S0 nil))
  (saturate* E R S ER E0 R0 S0))

(defun reset-wit (E R S)
  (loop for i in E do (setf (POL-wit i) nil))
  (loop for i in R do (setf (POL-wit i) nil))
  (loop for i in S do (setf (POL-wit i) nil)))

(defun saturate* (E R S ER E0 R0 S0)
  (cond (E
	 (multiple-value-bind (st E1 R1 S1)
	 	(decideDynamicE (car E) E0 R0 S0)
	   (if (eq st 'inconsistent) (values nil (get-witness E1))
		(saturate* (cdr E) R S ER E1 R1 S1))))
	(R
	 (multiple-value-bind (st E1 R1 S1)
	 	(decideDynamicR (car R) E0 R0 S0)
	   (if (eq st 'inconsistent) (values nil (get-witness E1))
		(saturate* nil (cdr R) S ER E1 R1 S1))))
	(S
	 (multiple-value-bind (st E1 R1 S1)
	 	(decideDynamicS (car S) E0 R0 S0)
	   (if (eq st 'inconsistent) (values nil (get-witness E1))
		(saturate* nil nil (cdr S) ER E1 R1 S1))))
	(ER
	 (multiple-value-bind (st E1 R1 S1)
	 	(case (cadr ER)
	 	    (> (decideDynamicR (car ER) E0 R0 S0))
		    (= (decideDynamicE (car ER) E0 R0 S0))
		    (t (decideDynamicS (car ER) E0 R0 S0)))
	   (if (eq st 'inconsistent) (values nil (get-witness E1))
		(saturate* nil nil nil (cddr ER) E1 R1 S1))))
	(t (values 'satisfiable E0 R0 S0))))

;; Add a new equation to a persisting E,R sets. Optimize later.
;; Return value: status, E, R, S, witness
(defun decideDynamicE (eqn E R &optional (S nil))
  (cond ((null eqn) (values 'satisfiable E R S nil))
	((prep:polyrepConstant? eqn) (values 'inconsistent (make-witness eqn '=)))
	(t
  (let* ((eqn1 (normalizePoly eqn)) 	;; (prep:polyrepDividePolyCst eqn (caar eqn)) CHECK
	 (eqn2 (make-POL :pol eqn1 :wit nil :op '=))
	 (*E* (cons eqn2 E))
	 (*R* R) (*S* S))
    (declare (special *E* *R* *S*))
    (if (linearPoly? eqn) (set-optimize-flag nil) (set-optimize-flag t))
    (decideGeneralDynamicE eqn2 E R S)))))

;; Add a new inequation to a persisting E,R sets.
(defun decideDynamicR (rule E R &optional (S nil))
  (cond ((null rule) (values 'inconsistent (make-witness rule '>)))
	((prep:polyrepConstant? rule) 
	 (if (<= (caar rule) 0) (values 'inconsistent (make-witness rule '>))
	     (values 'satisfiable E R S)))
	(t
  (let* ((rule1 (normalizePoly rule))
	 (rule2 (make-POL :pol rule1 :wit nil :op '>))
	 (*E* E) (*S* S)
	 (*R* (cons rule2 R)))
    (declare (special *E* *R* *S*))
    (if (linearPoly? rule) (set-optimize-flag nil) (set-optimize-flag t))
    (decideGeneralDynamicR rule2 E R S)))))

;; Add a new inequation to a persisting E,R sets.
(defun decideDynamicS (rule E R S)
  (cond ((null rule) (values 'satisfiable E R S))
	((prep:polyrepConstant? rule) 
	 (if (< (caar rule) 0) (values 'inconsistent (make-witness rule '>=))
	     (values 'satisfiable E R S)))
	(t
  (let* ((rule1 (normalizePoly rule))
	 (rule2 (make-POL :pol rule1 :wit nil :op '>=))
	 (*E* E) (*R* R)
	 (*S* (cons rule2 S)))
    (declare (special *E* *R* *S*))
    (if (linearPoly? rule) (set-optimize-flag nil) (set-optimize-flag t))
    (decideGeneralDynamicS rule2 E R S)))))
;; ----------------------------------------------------------------------------

(defmacro test-and-return (funname R op)
  `(multiple-value-bind (newR wit) (consistencyCheck ,R ,op)
     (if (eq newR 'inconsistent) (return-from ,funname (values newR wit)))
     newR))

;; ----------------------------------------------------------------------------
;; Deciding the nonlinear fragment:Generalized Fourier-Motzkin + Simplification
;; Return value: newE witness
;; ----------------------------------------------------------------------------
(defun decideGeneralDynamicE (eqn E R S)
  (print-debug 4 t "=======================================================================~%")
  (print-debug 4 t "ASSERTING ~A = 0~%" (prep:polyrepPrint (POL-pol eqn)))
  (print-debug 4 t "=======================================================================~%")
  (multiple-value-bind (newE oldE) (dynamicGrobner eqn E)
    (if (eq newE 'inconsistent) (return-from decideGeneralDynamicE (values newE oldE)))
    (print-debug 1 t "New eqn is ~A~%" (prep:polyrepPrint (POL-pol eqn)))
    (print-debug 1 t "OLD GB is ~A~%" (dpPrint E))
    (print-debug 1 t "New GB is ~% ~A~% ~A~%" (dpPrint newE) (dpPrint oldE))
  (let* ((newE (test-and-return decideGeneralDynamicE newE '=))
    	 (gb (sortByLPOAndMerge newE oldE)))
    (multiple-value-bind (st R1 R2) (dynamicSimplifyRbyErec gb R nil '>)
      (if (eq st 'inconsistent) (return-from decideGeneralDynamicE (values st R1)))
      (multiple-value-bind (st S1 S2) (dynamicSimplifyRbyErec gb S nil '>=)
    	(if (eq st 'inconsistent) (return-from decideGeneralDynamicE (values st R1)))
	(let* ((Einv (mapcar #'(lambda(x) (make-POL :pol (prep:polyrepNegativePoly (POL-pol x)) :wit (list x) :op '=)) newE))
	       (RE0 (append R1 R2))
	       (RE1 (chainPartialAllWith1 newE RE0 '> nil))
	       (RE2 (test-and-return decideGeneralDynamicE RE1 '>))
	       (RE3 (chainPartialAllWith1 Einv RE0 '> RE2))
	       (RE4 (test-and-return decideGeneralDynamicE RE3 '>))
	       (SE0 (append S1 S2))
	       (SE1 (chainPartialAllWith newE SE0 '>= nil))
	       (SE2 (test-and-return decideGeneralDynamicE SE1 '>=))
	       (SE3 (chainPartialAllWith Einv SE0 '>= SE2))
	       (SE4 (test-and-return decideGeneralDynamicE SE3 '>=)))
	 (if (or (> (length RE4) 30) (> (length SE4) 30))
	     (values 'satisfiable gb (nconc R1 R2) (nconc S1 S2))
	     (multiple-value-bind (st EF RF SF) 
			(decideNonLinearInequalitiesModE gb (nconc R1 RE4) R2 
				(nconc S1 SE4) S2 nil gb R S);;PASSING DEFAULT!!
	       (if (or (> (length RF) 50) (> (length SF) 50)) 
		   (print-debug 4 t "SIZE IS ~A and ~A.~%" (length RF) (length SF)))
	       (values st EF RF SF)))))))))

(defun decideGeneralDynamicR (rule E R S)
  (print-debug 4 t "=======================================================================~%")
  (print-debug 4 t "ASSERTING ~A > 0~%" (prep:polyrepPrint (POL-pol rule)))
  (print-debug 4 t "=======================================================================~%")
  (multiple-value-bind (st R1 R2) (dynamicSimplifyRbyErec E (list rule) nil '>)
    (if (eq st 'inconsistent) (return-from decideGeneralDynamicR (values st R1)))
  (multiple-value-bind (st r3) (getUpperBound (car (append R1 R2)) R S)
    (if (eq st 'inconsistent) (return-from decideGeneralDynamicR (values st r3)))
  (multiple-value-bind (st E3 R3 S3) (if st (decideGeneralDynamicR* (list r3) E R S) (values nil E R S))
    ;(if (and r3 (> (length (POL-pol rule)) 3) (not (linearPoly? (POL-pol rule))) (linearPoly? (POL-pol r3)) (not (eq st 'inconsistent)))
	;(progn (loop for i in R3 do (format t "~A > 0~%" (prep:polyrepPrint (POL-pol i)))) (break)))
    (if (eq st 'inconsistent) (return-from decideGeneralDynamicR (values st E3)))
    (decideGeneralDynamicR* (append R1 R2) E3 R3 S3)))))

(defun decideGeneralDynamicR* (newRule1 E R S)
  (if (and newRule1 (linearPoly? (POL-pol (car newRule1)))) (set-optimize-flag nil) (set-optimize-flag t))
  (let* ((RE1 (chainPartialAllWith1 E newRule1 '> nil))
	 (RE2 (test-and-return decideGeneralDynamicR* RE1 '>))
	 (Einv (mapcar #'(lambda(x) (make-POL :pol (prep:polyrepNegativePoly (POL-pol x)) :wit (POL-wit x) :op '=)) E))
	 (RE3 (chainPartialAllWith1 Einv newRule1 '> RE2))
	 (RE4 (test-and-return decideGeneralDynamicR* RE3 '>)))
    (if (> (length RE4) 30) 
	(progn (print-debug 2 t "VERY IMME TERMINATION~%") (values 'satisfiable E (nconc newRule1 R) S))
	(multiple-value-bind (st EF RF SF) (decideNonLinearInequalitiesModE E
			  (nconc RE4 newRule1) R nil S nil E (cons (car newRule1) R) S);;DEFAULTSpassed
	  (if (or (> (length RF) 50) (> (length SF) 50))
	      (print-debug 4 t "SIZE IS ~A and ~A.~%" (length RF) (length SF)))
	  (values st EF RF SF)))))

(defun decideGeneralDynamicS (rule E R S)
  (print-debug 4 t "=======================================================================~%")
  (print-debug 4 t "ASSERTING ~A >= 0~%" (prep:polyrepPrint (POL-pol rule)))
  (print-debug 4 t "=======================================================================~%")
  (multiple-value-bind (st S1 S2) (dynamicSimplifyRbyErec E (list rule) nil '>=)
    (if (eq st 'inconsistent) (return-from decideGeneralDynamicS (values st S1)))
    (let* ((newRule1 (nconc S1 S2))
	   (SE1 (chainPartialAllWith E newRule1 '>= nil))
	   (SE2 (test-and-return decideGeneralDynamicS SE1 '>=))
	   (Einv (mapcar #'(lambda(x) (make-POL :pol (prep:polyrepNegativePoly (POL-pol x)) :wit (POL-wit x) :op '=)) E))
	   (SE3 (chainPartialAllWith Einv newRule1 '>= SE2))
	   (SE4 (test-and-return decideGeneralDynamicS SE3 '>=)))
      (if (> (length SE4) 30) (values 'satisfiable E (nconc newRule1 R) S)
	  (multiple-value-bind (st EF RF SF) (decideNonLinearInequalitiesModE E nil R 
				(nconc SE4 newRule1) S nil E R (cons rule S));;DEFAULTSpassed
	    (if (or (> (length RF) 40) (> (length SF) 40))
		(print-debug 4 t "SIZE IS ~A and ~A.~%" (length RF) (length SF)))
	    (values st EF RF SF))))))
;; ----------------------------------------------------------------------------

;; ----------------------------------------------------------------------------
;; Fixed-point over decideNonLinearFourierMotzkin1 mod E.
;; Returns result after 1-iteration if fixpoint doesn't terminate!
;; ----------------------------------------------------------------------------
(defun decideNonLinearInequalitiesModE (E newR oldR newS oldS flag &optional (E0 nil) (R0 nil) (S0 nil) (n 0))
  (print-debug 3 t "---------Iteration ~A-----------------------------------------~%" n)
  (multiple-value-bind (st newR1 oldR1 newS1 oldS1)
	(decideNonLinearInequalitiesModE1 E newR oldR newS oldS)
    (print-debug 3 t "newR1=~A~%" (dpPrint newR1))
    (print-debug 3 t "newS1=~A~%" (dpPrint newS1))
    (print-debug 3 t "oldR1=~A~%" (dpPrint oldR1))
    (print-debug 3 t "oldS1=~A~%" (dpPrint oldS1))
    (print-debug 3 t "---------Iteration ~A END-------------------------------------~%" n)
    (cond ((eq st 'inconsistent)
	   (values st newR1))				;; newR1 is the WITNESS
	  ((and (null newR1) (null newS1))
	   (print-debug 3 t "GIVING up at ~%E=~A~%,newR=~A~%oldR=~A~%" (dpPrint E) (dpPrint newR) (dpPrint oldR))
	   (print-debug 5 t "T")	;; Termination
	   (values st E oldR1 oldS1))
	  ((and (null E0) (null R0) (null S0)
		(or (> (+ (length newR1) (length oldR1)) 20)
		    (> (+ (length newS1) (length oldS1)) 20)))
	   (print-debug 5 t "?")	;; What?
	   (values st E (append newR oldR) (append newS oldS)))
	  ((> n 50) (print-debug 5 t "N") (values st E0 R0 S0))	;; Nontermination
	  ((or (not flag) (and (null E0) (null R0) (null S0)))
	   (decideNonLinearInequalitiesModE E newR1 oldR1 newS1 oldS1 t E 
		(append newR oldR) (append newS oldS) (+ n 1)))	;; update E0 R0: newR2 oldR2!!
	  ((or (> (+ (length newR1) (length oldR1)) 75)		;; used to be 35
	       (> (+ (length newS1) (length oldS1)) 55))
	   (print-debug 5 t "F")	;; Forced termination
	   (values st E0 R0 S0))	;; return values after 1-iteration
	  (t
	   (decideNonLinearInequalitiesModE E newR1 oldR1 newS1 oldS1 t E0 R0 S0 (+ n 1))))))

;; One-step wrapper over decideNonLinearFourierMotzkin1 mod E.
(defun decideNonLinearInequalitiesModE1 (E newR oldR newS oldS)
  (print-debug 2 t "ModE1 called with ~%E=~A~%newR=~A~%oldR=~A~%" (dpPrint E) (dpPrint newR) (dpPrint oldR))
  (multiple-value-bind (st newNewR newOldR)
	(normalizeAndRemoveRedundant E newR oldR '>)
    (if (eq st 'inconsistent) (return-from decideNonLinearInequalitiesModE1 (values st newNewR)))
    (multiple-value-bind (st newNewS newOldS)
	(normalizeAndRemoveRedundant E newS oldS '>=)
      (if (eq st 'inconsistent) (return-from decideNonLinearInequalitiesModE1 (values st newNewS)))
      (print-debug 2 t "After redundandy:~%newR=~A~%" (dpPrint newNewR))
      (decideNonLinearFourierMotzkin1 newNewR newOldR newNewS newOldS))))

;; One step of Non-Linear FM: Return-value (status, new, old, deleted!)
(defun decideNonLinearFourierMotzkin1 (R oldR S oldS)
  (if (and (null R) (null S))
      (let ((newR (test-and-return decideNonLinearFourierMotzkin1 oldR '>))
            (newS (test-and-return decideNonLinearFourierMotzkin1 oldS '>=)))
	(values 'satisfiable nil newR nil newS))
      (let* ((p  (maxOfList (append R S) #'greater? :key #'POL-pol))
             (mu (cdar p)))
    	(multiple-value-bind (R2 R3 R4 R5) (splitOnLeadingPP oldR mu)
    	(multiple-value-bind (R6 R7 R8 R9) (splitOnLeadingPP R mu)
    	(multiple-value-bind (S2 S3 S4 S5) (splitOnLeadingPP oldS mu)
    	(multiple-value-bind (S6 S7 S8 S9) (splitOnLeadingPP S mu)
	  (assert (null R6))	;; NOT(mu properly divides leadingPP in R)
	  (assert (null S6))	;; NOT(mu properly divides leadingPP in R)
      	  (let* ((R10 (test-and-return decideNonLinearFourierMotzkin1 (chainAllWith R3 R7) '>))
		 (R11 (test-and-return decideNonLinearFourierMotzkin1 (chainAllPairs R7 '>) '>))
		 (R12 (test-and-return decideNonLinearFourierMotzkin1 (chainPartialAllWith R4 R7) '>))
		 (R13 (test-and-return decideNonLinearFourierMotzkin1 (chainPartialAllWith R8 R7) '>))
		 (R14 (test-and-return decideNonLinearFourierMotzkin1 (chainPartialAllWith R7 R2) '>))
		 (R15 (test-and-return decideNonLinearFourierMotzkin1 (chainAllWith S3 R7) '>))
		 (R16 (test-and-return decideNonLinearFourierMotzkin1 (chainAllWith S7 R7) '>))
		 (R17 (test-and-return decideNonLinearFourierMotzkin1 (chainAllWith S4 R7) '>))
		 (R18 (test-and-return decideNonLinearFourierMotzkin1 (chainAllWith S8 R7) '>))
		 (R19 (test-and-return decideNonLinearFourierMotzkin1 (chainAllWith R7 S2) '>))
		 (R20 (test-and-return decideNonLinearFourierMotzkin1 (chainAllWith R3 S7) '>))
		 (R21 (test-and-return decideNonLinearFourierMotzkin1 (chainPartialAllWith1 R4 S7) '>))
		 (R22 (test-and-return decideNonLinearFourierMotzkin1 (chainPartialAllWith1 R8 S7) '>))
		 (R23 (test-and-return decideNonLinearFourierMotzkin1 (chainPartialAllWith1 S7 R2) '>))
		 (S10 (test-and-return decideNonLinearFourierMotzkin1 (chainAllWith S3 S7 '>=) '>=))
		 (S11 (test-and-return decideNonLinearFourierMotzkin1 (chainAllPairs S7 '>=) '>=))
		 (S12 (test-and-return decideNonLinearFourierMotzkin1 (chainPartialAllWith S4 S7 '>=) '>=))
		 (S13 (test-and-return decideNonLinearFourierMotzkin1 (chainPartialAllWith S8 S7 '>=) '>=))
		 (S14 (test-and-return decideNonLinearFourierMotzkin1 (chainPartialAllWith S7 S2 '>=) '>=))
		 (newR (nconc R10 R11 R12 R13 R14 R8 R9 R15 R16 R17 R18 R19 R20 R21 R22 R23))
		 (newS (nconc S10 S11 S12 S13 S14 S8 S9))
		 (oldR (nconc R3 R4 R5 R2 R7))
		 (oldS (nconc S3 S4 S5 S2 S7))) 	;; maybe a subtle bug here.
	      (values 'satisfiable newR oldR newS oldS)))))))))

;; return (R1 R2 R3 R4 status), R1=mu divides, R2=equal, R3=divide mu, R4=rest.
(defun splitOnLeadingPP (R mu &key (key nil) (test nil))
  (splitOnLeadingPP* R mu key test nil nil nil nil))

;; test: if r\in R has same leading PP as mu, AND it satisfies test, then put it in R2
;; key: if r\in R satisfies above AND key is absent in r, then terminate early!
(defun splitOnLeadingPP* (R mu key test R1 R2 R3 R4)
  (if (null R) (values R1 R2 R3 R4)
      (let* ((p1 (car R)) (p1p (POL-pol p1)))
	(if (or (null p1p) (null (cdar p1p)))
	    (splitOnLeadingPP* (cdr R) mu key test R1 R2 R3 R4)
	    (let* ((mu1 (cdar p1p))
		   (mu1Bymu (prep:polyrepDividePP mu1 mu))
		   (muBymu1 (prep:polyrepDividePP mu mu1)))
	      (cond ((and (eq mu1Bymu 0) (eq muBymu1 0))
	       	     (splitOnLeadingPP* (cdr R) mu key test R1 R2 R3 (cons p1 R4)))
		    ((eq mu1Bymu nil)
		     (if (and test (funcall test p1p))
			 (if (and key (not (funcall key p1p)))
			     (return-from splitOnLeadingPP* (values R1 (cons p1 R2) R3 R4))
	       	     	     (splitOnLeadingPP* (cdr R) mu key test R1 (cons p1 R2) R3 R4))
			 (if (null test)
	       	     	     (splitOnLeadingPP* (cdr R) mu key test R1 (cons p1 R2) R3 R4)
	       	     	     (splitOnLeadingPP* (cdr R) mu key test R1 R2 R3 R4))))
		    ((not (eq mu1Bymu 0))
	       	     (splitOnLeadingPP* (cdr R) mu key test (cons p1 R1) R2 R3 R4))
		    (t
	       	     (splitOnLeadingPP* (cdr R) mu key test R1 R2 (cons p1 R3) R4))))))))
;; ----------------------------------------------------------------------------

;; ----------------------------------------------------------------------------
;; Compute the Grobner Basis for the set E of polynomial equations.
;; This seems to be a fairly interestingly correct implementation.
;; ----------------------------------------------------------------------------
;; Assuming: E is sorted.
;; NONTRIVIAL RULE: swap eqn by an eqn in E with the same LHS but
;; simpler RHS.
(defun dynamicGrobner (eqn E)
  (let* ((e1 (POL-pol eqn))
	 (eqn1 (loop for i in E
		if (eq (prep:polyrepComparePP (cdar (POL-pol i)) (cdar e1)) prep:'equal)
		return i)))
    (cond ((null eqn1)
	   (print-debug 1 t "Eqn1 is null.~%")
	   (grobner1* (list eqn) nil E))
    	  ((eq (prep:polyrepComparePP (cdadr e1) (cdadr (POL-pol eqn1))) prep:'less)
	   ;; (setf (nth index E) eqn)
	   ;; (print-debug 1 t "Calling Grobner with E~%~A~%" (dpPrint E))
	   ;; (grobner1* (list eqn1) nil E)
	   (grobner1* (list eqn1) nil (substitute eqn eqn1 E)))
	  (t (grobner1* (list eqn) nil E)))))
 
;; res is a sorted GB and E are the equations to be added/processed.
;; foreach neweqn in E, collapse/superpose with all eqns in res.
;; collapse and superpose completely
(defun grobner1* (unprocNew &optional (procNew nil) (old nil))
  (cond ((null unprocNew) (values procNew old))
	((and (null procNew) (null old))
	 (grobner1* (cdr unprocNew) (list (car unprocNew)) nil))
	(t
	 (multiple-value-bind (st newE oldE)	;; collapse new eqn
		(dynamicSimplifyRbyErec (append procNew old) 
					(list (car unprocNew)) nil '=)
	   (if (eq st 'inconsistent)
	       (values st newE)		;; newE = witness
	       (let ((neweqn (car (nconc newE oldE))))
	         (if neweqn
	             (multiple-value-bind (st newOld oldOld)
				(grobner2* old neweqn)
	   	     (if (eq st 'inconsistent) (values st newOld)
	             (multiple-value-bind (st newProcNew oldProcNew)
				(grobner2* procNew neweqn)
	   	       (if (eq st 'inconsistent) (values st newProcNew)
	       	           (grobner1* 
				(sortByLPOAndMerge (nconc newProcNew newOld) 
						   (cdr unprocNew))
				(sortByLPOAndMerge (list neweqn) oldProcNew)
				oldOld)))))
	       	     (grobner1* (cdr unprocNew) procNew old))))))))

;; Collapse "processed"-list (a GB) by the given (fully-reduced) polynomial q.
;; Return value: (new collapsed polys + new superpositions, old unchanged pols)
(defun grobner2* (processed q &optional (new nil) (old nil))
  (if (null processed) 
      (values 'satisfiable (nreverse new) (nreverse old))
      (let* ((newp0 (collapseEE (car processed) q))
	     (newp1 (POL-pol newp0)))
	(cond ((null newp1)
	       (let ((newp3 (superposition (car processed) q)))
	         (if (null (POL-pol newp3))
	             (grobner2* (cdr processed) q new 
				(cons (car processed) old))
	             (grobner2* (cdr processed) q 
				(cons newp3 new) 
				(cons (car processed) old)))))
	      ((and (null (cdr newp1)) 
		    (null (cdar newp1))
		    (eq (caar newp1) 0))
	       (grobner2* (cdr processed) q new old))
	      ((and (null (cdr newp1)) 
		    (null (cdar newp1)))
	       (values 'inconsistent (POL-wit newp0)))
	      (t (grobner2* (cdr processed) q (cons newp0 new) old))))))

;; ----------------------------------------------------------------------------
;; Operations related to Merging of Sorted Lists
;; ----------------------------------------------------------------------------
(defun sortByLPOAndMerge (unsorted sorted)
  (mergeSortedLists* (sort unsorted #'greater? :key #'POL-pol) sorted #'greater? nil :key #'POL-pol))

(defun mergeSortedLists* (l1 l2 fn res &key (key nil))
  (cond ((null l1) (nconc (nreverse res) l2))
	((null l2) (nconc (nreverse res) l1)) 
	(t (let ((p1 (car l1))
		 (p2 (car l2)))
	     (if (funcall fn (funcall key p1) (funcall key p2))
		 (mergeSortedLists* (cdr l1) l2 fn (cons p1 res) :key key)
		 (mergeSortedLists* l1 (cdr l2) fn (cons p2 res) :key key))))))
;; ----------------------------------------------------------------------------

;; ----------------------------------------------------------------------------
;; Simplify plist using the polynomial p (i.e. p=0) as a rewrite rule
;; Return a new plist. Note: Simplify not called recursively
;; ----------------------------------------------------------------------------
;; FULLY Simplify R by E, returned (status newR oldR)
;; E should be sorted
(defun dynamicSimplifyRbyErec (E R new op)
  (if (null E) (values 'satisfiable new R)
      (multiple-value-bind (R1 R2) (dynamicSimplify (car E) R) 
      (multiple-value-bind (R3 R4) (dynamicSimplify (car E) new) 
	(let* ((newR1 (test-and-return dynamicSimplifyRbyErec (simplifyRecEqnR (car E) R1) op))
	       (newR3 (test-and-return dynamicSimplifyRbyErec (simplifyRecEqnR (car E) R3) op)))
      	  (dynamicSimplifyRbyErec (cdr E) R2 (nconc newR1 newR3 R4) op))))))

;; FULLY Simplify R by eqn, returned (newR oldR)
(defun simplifyRecEqnR (eqn R &optional (res nil))
  (multiple-value-bind (R1 R2) (dynamicSimplify eqn R)
    (if (null R1) (nconc R res)
        (simplifyRecEqnR eqn R1 (nconc R2 res)))))

;; Dynamic variants: Simplify each poly in plist by p = 0
(defun dynamicSimplify (p plist)
  (cond ((null plist)
	 (values nil nil))
	((null p)
	 (values nil plist))
	(t (dynamicSimplify* p plist))))

;; Simplify each poly in plist by p = 0
(defun dynamicSimplify* (p plist)
  (let* ((c0s0 (car (POL-pol p)))
	 (c0 (* (car c0s0) -1))		;; -1 for moving cisi to RHS!!
	 (cisi (prep:polyrepDividePolyCst (cdr (POL-pol p)) c0))
	 (s0 (cdr c0s0)))
    (dynamicSimplifyList* s0 cisi plist (list p))))	;; (list p) is the WITNESS

;; Replace s0 by cisi in each poly in plist: 
;; Return only (changed old) pair
(defun dynamicSimplifyList* (s0 cisi plist wit &optional (new nil) (old nil))
  (if (null plist) (values new old)
      (let* ((p (car plist))
	     (newp (simplify-poly* s0 cisi (POL-pol p))))
        (if (not (eq newp 0)) (print-debug 1 t "Simplified~A to~% ~A~%" p newp))
	(if (eq newp 0)
	    (dynamicSimplifyList* s0 cisi (cdr plist) wit new (cons p old))
	    (let* ((newp1 (normalizePoly newp))
		   (newp2 (combine-wit wit p))
		   (newp3 (make-POL :pol newp1 :wit newp2 :op (POL-op p))))
	      (dynamicSimplifyList* s0 cisi (cdr plist) wit (cons newp3 new) old))))))

;; NO WITNESS HANDLING IN THE FOLLOWING.
;; Replace s0 by cisi in each poly in plist
(defun simplify-poly* (s0 cisi poly)
  (if (null poly) 0
      (let ((poly1 (simplify-mono* s0 cisi (car poly)))
	    (poly2 (simplify-poly* s0 cisi (cdr poly))))
        (cond ((and (eq poly1 0) (eq poly2 0))
	       0)
	      ((eq poly1 0)
	       (cons (car poly) poly2))	;; ordering guarantees this
	      ((eq poly2 0)
	       (prep:polyrepAddPoly poly1 (cdr poly)))
	      (t 
	       (prep:polyrepAddPoly poly1 poly2))))))

(defun simplify-mono* (s0 cisi mono)
  (let* ((d0 (car mono))
	 (t0 (cdr mono))
	 (newt0 (simplify-pp* s0 cisi t0)))
    (if (eq newt0 0)
	0
	(prep:polyrepMultiplyCstPoly d0 newt0))))

(defun simplify-pp* (s0 cisi t0)
  (let ((t0bys0 (prep:polyrepDividePP t0 s0)))
    (if (eq t0bys0 0)
	0
	(prep:polyrepMultiplyMonoPoly (cons 1 t0bys0) cisi))))
;; ----------------------------------------------------------------------------

;; ----------------------------------------------------------------------------
;; Chaining: Returns nil or the chained polynomial
;; Consistency of result not checked.
;; ----------------------------------------------------------------------------
;; chainingStrict for all pairs of R.
(defun chainAllPairs (R &optional (op '>) (res nil))
  (if (null R) res
      (multiple-value-bind (res1 wit1) (chainOneAll* (car R) (cdr R) op)
	(if (eq res1 -1) wit1
	    (chainAllPairs (cdr R) op (nconc res1 res))))))

;; chainingStrict for all pairs (p1, p2), p1 in R, p2 in R1.
(defun chainAllWith (R R1 &optional (op '>) (res nil))
  (if (null R1) res
      (multiple-value-bind (res1 wit1) (chainOneAll* (car R1) R op)
	(if (eq res1 -1) wit1
	    (chainAllWith R (cdr R1) op (nconc res1 res))))))

(defun chainOneAll* (rule R op)
  (loop for i in R for j = (chainingStrict i rule) 
	if (and (not (numberp (POL-pol j))) (or (eq op '>) (POL-pol j))) collect j
	if (and (eq op '>) (null (POL-pol j))) return (values -1 (list j))))

;; chainPartial (r1, r2), with r3 chosen from *R*+*S*
(defun chainPartialAllWith (R1 R2 &optional (op '>) (res nil))
  (declare (special *R* *S*))
    (chainPartialAllWith* R1 R2 (append *R* *S*) op res))

;; chainPartial (r1, r2), with r3 chosen from *R*
(defun chainPartialAllWith1 (R1 R2 &optional (op '>) (res nil))
  (declare (special *R*))
  (chainPartialAllWith* R1 R2 *R* op res))

(defun chainPartialAllWith* (R1 R2 RS op res)
  (if (null R2) res
      (let ((rule (car R2)))
      (multiple-value-bind (res1 wit1)
	     (loop for i in R1 for j = (chainingPartial i rule RS)
		   if (and (not (numberp (POL-pol j))) (or (eq op '>) (POL-pol j)))
		   collect j
		   if (and (eq op '>) (null (POL-pol j))) return (values -1 (list j)))
	(if (eq res1 -1) wit1
	    (chainPartialAllWith* R1 (cdr R2) RS op (nconc res1 res)))))))
;; ----------------------------------------------------------------------------

;; ----------------------------------------------------------------------------
;; Superposition, ChainingStrict, ChainingPartial, Collapse
;; ----------------------------------------------------------------------------
;; Strict superposition, when leading PPs are equal.
(defun superposition (p q)
  (let ((cp (superposition* (POL-pol p) (POL-pol q))))
    (make-POL :pol cp :wit (combine-wit p q) :op '=)))
	
(defun superposition* (p q)
  (cond ((null p) nil)
	((null q) nil)
	((eq (prep:polyrepComparePP (cdar p) (cdar q)) prep:'equal)
	 (normalizePoly (prep:polyrepAddPoly 
		(prep:polyrepMultiplyCstPoly (caar q) (cdr p)) 
		(prep:polyrepMultiplyCstPoly (- 0 (caar p)) (cdr q)))))
	(t nil)))

;; Collapse first polynomial by the second one
(defun collapseEE (p q)
  (let ((cp (collapseEE* (POL-pol p) (POL-pol q))))
    (make-POL :pol cp :wit (combine-wit p q) :op '=)))

(defun collapseEE* (p q)
  (let* ((c0s0 (car p))
	 (d0t0 (car q))
	 (c0 (car c0s0))
	 (d0 (car d0t0))
	 (d0p c0)
	 (c0p (- 0 d0)))
    (multiple-value-bind (overlap? s0p t0p)
	(overlap-pp-pp (cdr c0s0) (cdr d0t0))
	(if (and overlap? (null s0p))
	    (let ((ans (prep:polyrepAddPoly 
			(prep:polyrepMultiplyCstPoly c0p (cdr p)) 
			(prep:polyrepMultiplyMonoPoly (cons d0p t0p) (cdr q)))))
	      (normalizePoly ans))
	    nil))))

;; Return value: POL s.t POL-pol = 0 IF no chaining inference, =nil/polyrep IF answer
(defun chainingStrict (p1 q1)
  (let ((ans (chainingStrict* (POL-pol p1) (POL-pol q1))))
    (make-POL :pol ans :wit (if (eq ans 0) nil (combine-wit p1 q1)))))

(defun chainingStrict* (p q)
    (cond ((null p) 0)
	  ((null q) 0)
	  ((and (> (caar p) 0) (> (caar q) 0)) 0)
	  ((and (< (caar p) 0) (< (caar q) 0)) 0)
	  (t (chainingStrict** p q))))

;; Leading PPs assumed equal. Chain for this special case.
(defun chainingStrict** (p q)
  (let ((d0p (abs (/ (caar p) (caar q)))))
    (normalizePoly (prep:polyrepAddPoly (cdr p) 
	(prep:polyrepMultiplyCstPoly d0p (cdr q))))))

;; ----------------------------------------------------------------------------
;; Chain p, q using some rule/rules in R.
;; ----------------------------------------------------------------------------
(defun chainingPartial (p q R)
  (multiple-value-bind (ans wit) (chainingPartial* (POL-pol p) (POL-pol q) R)
    (make-POL :pol ans :wit (if (eq ans 0) nil (combine-wit p q wit)))))

;; Return the rule(s) in R  used for purposes of chaining
(defun chainingPartial* (p q R)
  (cond ((null p) 0)
	((null q) 0)
	(t (chainingPartial** p q R))))

(defun chainingPartial** (p q R)
  (let* ((c0s0 (car p))
	 (d0t0 (car q))
	 (c0 (car c0s0))
	 (d0 (car d0t0))
	 (d0p (abs (/ c0 d0)))
	 (sgn (or (and (> c0 0) (< d0 0)) (and (< c0 0) (> d0 0)))))
    (multiple-value-bind (overlap? s0p t0p)
	(overlap-pp-pp (cdr c0s0) (cdr d0t0))
      (cond ((null overlap?) 0)
	    ((null s0p)
	     (multiple-value-bind (ans wit) (chainingPartial*** d0p t0p sgn q p R)
		(if (not (eq ans -1)) (print-debug 3 t "Chaining ~A and ~A yields ~A~%" (prep:polyrepPrint p) (prep:polyrepPrint q) (prep:polyrepPrint ans)))
	       (if (or (eq ans -1) (strongRedundant? d0 d0t0 ans)) 0
		   (values (normalizePoly ans) wit))))
	    ((null t0p)
	     (multiple-value-bind (ans wit) (chainingPartial*** (/ 1 d0p) s0p sgn p q R)
		(if (not (eq ans -1)) (print-debug 3 t "Chaining ~A and ~A yields ~A~%" (prep:polyrepPrint p) (prep:polyrepPrint q) (prep:polyrepPrint ans)))
	       (if (or (eq ans -1) (strongRedundant? c0 c0s0 ans)) 0
		   (values (normalizePoly ans) wit))))
	    (t 0))))) 	;; partial only!

(defun chainingPartial*** (c0p s0p sgn p q R)
  (multiple-value-bind (factor wit) (findRule2 c0p s0p sgn R)
    (if factor 
	(values (prep:polyrepAddPoly (prep:polyrepMultiplyPoly factor p) q) wit) 
	-1)))

(defun strongRedundant? (c0 c0s0 p)
  ;; (or (and (> (length p) (length p1)) (> (length p) (length p2))))
  (and (eq (prep:polyrepComparePP (cdar p) (cdr c0s0)) prep:'equal)
       (or (and (> c0 0) (> (caar p) 0))
	   (and (< c0 0) (< (caar p) 0)))))

(defun overlap-pp-pp (mu nu)
  (let ((order (prep:get-order)))
    (overlap-pp-pp* mu nu order nil nil nil)))

(defun overlap-pp-pp* (mu nu order yes mu1 nu1)
  (if (null order) 
      (values yes (nreverse mu1) (nreverse nu1))
      (let* ((x1 (car order))
	     (x1c1 (assoc x1 mu :test #'prep:var-equal?))
	     (x1c2 (assoc x1 nu :test #'prep:var-equal?))
	     (c1 (cdr x1c1))
	     (c2 (cdr x1c2)))
	(cond ((and (null c1) (null c2))
	       (overlap-pp-pp* mu nu (cdr order) yes mu1 nu1))
	      ((null c1)
	       (overlap-pp-pp* mu nu (cdr order) yes (cons x1c2 mu1) nu1))
	      ((null c2)
	       (overlap-pp-pp* mu nu (cdr order) yes mu1 (cons x1c1 nu1)))
	      ((= c1 c2)
	       (overlap-pp-pp* mu nu (cdr order) t mu1 nu1))
	      ((> c1 c2)
	       (overlap-pp-pp* mu nu (cdr order) t mu1
			(cons (cons (car x1c2) (- c1 c2)) nu1)))
	      ((< c1 c2)
	       (overlap-pp-pp* mu nu (cdr order) t 
			(cons (cons (car x1c2) (- c2 c1)) mu1) nu1))))))
;; ----------------------------------------------------------------------------

;; ----------------------------------------------------------------------------
;; normalizePoly: Make leading coefficient +1 or -1.
;; ----------------------------------------------------------------------------
(defun normalizePoly (p)
  (if (null p) p
      (let ((c0 (abs (caar p))))
	(if (eq c0 1) p (prep:polyrepDividePolyCst p c0)))))

(defun nnormalizePoly (p)
  (if (null p) p
      (let ((c0 (abs (caar p))))
	(if (not (eq c0 1)) (loop for c0s0 in p do (setf (car c0s0) (/ (car c0s0) c0))))
	p)))
;; ----------------------------------------------------------------------------

;; ----------------------------------------------------------------------------
;; Consistency Checks. Input could be inconsistent (chainPartialAllWith)
;; ----------------------------------------------------------------------------
(defun consistencyCheck (plist op)
  ;;(if (eq plist 'inconsistent) (return-from consistencyCheck 'inconsistent))
  (loop for i in plist 
	for j = (formula-status (POL-pol i) op)
	if (eq j 'satisfiable) collect i
	if (eq j 'inconsistent) return (values j (POL-wit i))))

;; formula-status : check if the atomic formula is not 0 > 0
;; Return value: 'valid, 'satisfiable, 'inconsistent
(defun formula-status (poly &optional (op '>))
  (cond ((null poly)
	 (if (eq op '>) 'inconsistent 'valid))
	((prep:polyrepConstant? poly)
	 (cond ((and (eq op '>) (> (caar poly) 0)) 'valid)
	       ((eq op '>) 'inconsistent)
	       ((and (eq op '=) (eq (caar poly) 0)) 'valid)
	       ((eq op '=) 'inconsistent)
	       ((and (eq op '>=) (>= (caar poly) 0)) 'valid)
	       (t 'inconsistent)))
	(t 'satisfiable)))
;; ----------------------------------------------------------------------------

;; ----------------------------------------------------------------------------
;; findRule: return the rule with this mu on LHS (with pos coef if sgn).
;; ----------------------------------------------------------------------------
;; Wrapper around findRule. Search for 2 rules whose Leading PP
;; product is equal to this rule...do this only when findRule fails.
(defun findRule2 (c0 s0 sgn R)
  (let ((ans (findRule2* s0 sgn R)))
    (if ans (values (generateCorrectPoly c0 ans) ans))))

(defun findRule2* (s0 sgn R)
  (multiple-value-bind (R1 R2 R3 R4) 
	(splitOnLeadingPP R s0 :key #'cddr :test #'(lambda(x) (compatibleSign? sgn '((1)) x)))
    (declare (ignore R1 R4))
    (if R2 
	(let ((ans (find-if-not #'cddr R2 :key #'POL-pol)))
	  (list (if ans ans (car R2))))
	(let ((ans (findRule2** s0 sgn 1 R3)))
	  (if ans (car ans) nil)))))

(defun findRule2** (s0 sgn n R)
  (let ((*n* n))
    (loop for i on R when (> *n* 0) if 
	  (loop for j in (cdr i) when (> *n* 0)
	        if (let ((p (POL-pol (car i))) (q (POL-pol j)))
		     (and (compatibleSign? sgn p q)
			  (compatiblePP? s0 p q)
			  (optimizedChoice? p q)))
		collect (progn (decf *n*) (list (car i) j)))
	  append it)))

(defun compatibleSign? (sgn p q)
  (or (and sgn (> (caar p) 0) (> (caar q) 0))
      (and sgn (< (caar p) 0) (< (caar q) 0))
      (and (not sgn) (> (caar p) 0) (< (caar q) 0))
      (and (not sgn) (< (caar p) 0) (> (caar q) 0))))

(defun compatiblePP? (s0 p q)
  (eq (prep:polyrepComparePP s0 (prep:polyrepMultiplyPP (cdar p) (cdar q))) prep:'equal))

(defun optimizedChoice? (p q)
  (not (and (get-optimize-flag) (or (cddr p) (cddr q)))))

(defun generateCorrectPoly (c0 ans)
  (let* ((p (POL-pol (car ans))) (q (if (cdr ans) (POL-pol (cadr ans))))
	 (p0 (if (cdr ans) (prep:polyrepMultiplyPoly p q) p))
	 (d0 (if (cdr ans) (* (caar p) (caar q)) (caar p))))
    (prep:polyrepMultiplyCstPoly (/ c0 (abs d0)) p0))) 

;; Return value: (num rule rest);  num = 2 if more than one match; 
;; rule = first match; rest = (R - match).
(defun findRuleParity (s0 parity len R &optional (rr nil) (res nil))
  (if (null R) (values 1 rr (nreverse res))
      (let* ((p0 (POL-pol (car R)))
	     (d0 (caar p0))
	     (t0 (cdar p0)))
	(if (and (or (and parity (> d0 0))
		     (and (null parity) (< d0 0))) 
		 (eq (prep:polyrepComparePP s0 t0) prep:'equal)
		 (>= len (length p0)))
	    (if rr (values 2 rr (nconc (nreverse res) R))
		   (findRuleParity s0 parity len (cdr R)
			(prep:polyrepMultiplyCstPoly (/ 1 (abs d0)) p0) res))
		    	;; (nconc (nreverse res) (cdr R))
	    (findRuleParity s0 parity len (cdr R) rr (cons (car R) res))))))
;; ----------------------------------------------------------------------------

;; ----------------------------------------------------------------------------
;; factorize-vars: Simplify atomic formula by factoring by GCD.
;; Return: new polynomial and the factored power-product.
;; ----------------------------------------------------------------------------
(defun factorize-vars (p)
  (if (null p)
      (values p nil)
      (let* ((hcf (get-gcd p (prep:get-order)))
	     (newpoly (if hcf (prep:polyrepDividePolyPP p hcf) p)))
	(values newpoly hcf))))

(defun get-gcd (p order &optional (res nil))
  (if (null p) (nreverse res)
      (let* ((x1 (car order))
	     (x1cis (loop for i in p collect (assoc x1 (cdr i) :test #'prep:var-equal?)))
	     (x1power (apply #'min 
		(cons 0 (loop for i in x1cis if i collect (cdr i)))))
	     (x1real (if (eq x1power 0) nil 
		(caar (loop for i in x1cis if i collect i)))))
	(if (eq x1power 0)
	    (get-gcd p (cdr order) res)
	    (get-gcd p (cdr order) (cons (cons x1real x1power) res))))))
;; ----------------------------------------------------------------------------

;; ----------------------------------------------------------------------------
;; maxOfList: return maximum element of list, use fn to compare.
;; ----------------------------------------------------------------------------
(defun maxOfList (plist fn &key (key nil))
  (assert (car plist))
  (maxOfList* (cdr plist) fn (funcall key (car plist)) key))

(defun maxOfList* (plist fn ans key)
  (if (null plist) ans
      (if (funcall fn (funcall key (car plist)) ans)
	  (maxOfList* (cdr plist) fn (funcall key (car plist)) key)
	  (maxOfList* (cdr plist) fn ans key))))
;; ----------------------------------------------------------------------------

;; ----------------------------------------------------------------------------
;; redundant: Is "poly op 0" redundant in E R ?
;; ----------------------------------------------------------------------------
(defun redundant (poly op R)
  (let ((pol (POL-pol poly)))
    (redundant* pol op R)))

(defun redundant* (pol op R)
  (let ((*unsound* t)
	(*heavy* (not (linearPoly? pol))))
    (declare (special *unsound* *heavy*))
    (redundant** pol op R 1)))

(defun redundant** (pol op R n)			;; redundant, n levels deep.
  (declare (special *unsound* *heavy*))		;; For termination: 0 > # > -1/10
  (let ((s0 (cdar pol))
	(rule0 (normalizePoly pol)))
    (multiple-value-bind (num rule1 others)
		(findRuleParity s0 (> (caar rule0) 0) (length rule0) R)
      (if (and *heavy* (> num 1)) (return-from redundant** (values t R)))	;; Severe optimization
      (let* ((rule2 (if rule1 (prep:polyrepAddPoly rule0 (prep:polyrepNegativePoly rule1))))
	     (status (if rule1 (formula-status rule2 op))))
	(cond ((not rule1) (values nil R))		;; rule0 = rule1 + rule2
	      ((null rule2) (values t R))
	      ((eq status 'valid) (values t R))
	      ((and *unsound* (eq status 'inconsistent) (> (caar rule2) (/ -1 10))) (values t R))
	      ((eq status 'inconsistent) (values nil others))
	      (t
	       (cond ((eq n 0) (values nil R))
		     ((redundant** rule2 op R (- n 1)) (values t R))
		     ; ((redundant** (prep:polyrepNegativePoly rule2) op R (- n 1)) (values nil others)) ;; NOT CORRECT!!!!
		     (t (values nil R)))))))))

(defun removeRedundant2way1 (polylist op &optional (res nil))
  (if (null polylist) (nreverse res)
      (multiple-value-bind (yes? newR) 
		(redundant (car polylist) '> (cdr polylist))
	(print-debug 1 t "2way1: redundant on ~%~A returns~%~A,~A~%" polylist yes? newR)
	(if yes? 
	    (removeRedundant2way1 (cdr polylist) op res)
	    (removeRedundant2way1 newR op (cons (car polylist) res))))))

(defun removeRedundant2way2 (polylist op R &optional (res nil))
  (if (null polylist) (values (nreverse res) R)
      (multiple-value-bind (yes? newR) 
		(redundant (car polylist) '> R)
	(if yes? 
	    (removeRedundant2way2 (cdr polylist) op R res)
	    (removeRedundant2way2 (cdr polylist) op newR
				  (cons (car polylist) res))))))

(defun removeRedundant2way (polylist op R)
  (let ((newR1 (removeRedundant2way1 polylist op)))
    (print-debug 1 t "RemoveRedundant2way: Changed ~%~A to ~%~A~%" polylist newR1)
    (removeRedundant2way2 newR1 op R)))

(defun normalizeAndRemoveRedundant (E newR R op)
  (multiple-value-bind (status newR2 newR3)
	(dynamicSimplifyRbyErec E newR nil op)
    (if (eq status 'inconsistent) (values 'inconsistent newR2)
  	(multiple-value-bind (newNewR newOldR)
		(removeRedundant2way (nconc newR2 newR3) op R)
	  (values 'satisfiable newNewR newOldR)))))
;; ----------------------------------------------------------------------------

;; ----------------------------------------------------------------------------
;; check-invariant-new and check-equivalence-new: Use the new
;; decision procedures. Check if (E0,R0) |- fml1 => fml2
;; ----------------------------------------------------------------------------
(defun check-invariant-new (fml1 op1 fml2 op2 &optional (E0 nil) (R0 nil) (S0 nil))
  (multiple-value-bind (status1 E1 R1 S1)
	(cond ((eq op1 '=) (decideDynamicE fml1 E0 R0 S0))
	      ((eq op1 '>) (decideDynamicR fml1 E0 R0 S0))
	      ((eq op1 '<) (decideDynamicR (prep:polyrepNegativePoly fml1) E0 R0 S0))
	      ((eq op1 '>=) (decideDynamicS fml1 E0 R0 S0))
	      ((eq op1 '<=) (decideDynamicS (prep:polyrepNegativePoly fml1) E0 R0 S0))
	      (t ;(sal-error op1 "PANIC: check-invariant-new cannot handle~A~%" op1) 
		nil))
    (if (eq status1 'inconsistent) t   ;; SPECIAL SPECIAL CASE
	(cond ((eq op2 '<>) 
	       (let ((s3 (decideDynamicE fml2 E1 R1 S1)))
		 (if (eq s3 'satisfiable) nil t)))
	      ((eq op2 '>=) 
	       (let ((s3 (decideDynamicR (prep:polyrepNegativePoly fml2) E1 R1 S1)))
		 (if (eq s3 'satisfiable) nil t)))
	      ((eq op2 '<=)
	       (let ((s3 (decideDynamicR fml2 E1 R1 S1)))
		 (if (eq s3 'satisfiable) nil t)))
	      ((eq op2 '=) 
	       (let ((s2 (decideDynamicR fml2 E1 R1 S1)))
		 (if (eq s2 'satisfiable) nil
	       	 (let ((s3 (decideDynamicR (prep:polyrepNegativePoly fml2) E1 R1)))
		 (if (eq s3 'satisfiable) nil t)))))
	      ((eq op2 '>) 
	       (let ((s3 (decideDynamicS (prep:polyrepNegativePoly fml2) E1 R1 S1)))
		 (if (eq s3 'satisfiable) nil t)))
	       ;(let ((s2 (decideDynamicE fml2 E1 R1 S1)))
		 ;(if (eq s2 'satisfiable) nil
	       	 ;(let ((s3 (decideDynamicR (prep:polyrepNegativePoly fml2) E1 R1)))
		 ;(if (eq s3 'satisfiable) nil t))))
	      ((eq op2 '<)
	       (let ((s3 (decideDynamicS fml2 E1 R1 S1)))
		 (if (eq s3 'satisfiable) nil t)))
	       ;(let ((s2 (decideDynamicE fml2 E1 R1 S1)))
		 ;(if (eq s2 'satisfiable) nil
	       	     ;(let ((s3 (decideDynamicR fml2 E1 R1 S1)))
		 	;(if (eq s3 'satisfiable) nil t))))
	      (t (print-debug 5 t "PANIC: check-invariant-new cannot handle~A~%" op2)
	 	(break)
		nil)))))

(defun check-equivalence-new (fml1 op1 fml2 op2 &optional (E0 nil) (R0 nil) (S0 nil))
  (if (check-invariant-new fml1 op1 fml2 op2 E0 R0 S0)
      (if (check-invariant-new fml2 op2 fml1 op1 E0 R0 S0) t nil) nil))
;; ----------------------------------------------------------------------------

;; ----------------------------------------------------------------------------
;; Return everything in oldR, plus unique things from newR
;; ----------------------------------------------------------------------------
(defun get-optimize-flag ()
  *optimize*)

(defun set-optimize-flag (flag)
  (setf *optimize* flag))

(defun set-debug-level (n)
  (setf *dlevel* n))

(defun greater? (x y)
  (case (prep:polyrepComparePoly x y)
	(prep:greater t) (t nil)))
;; ----------------------------------------------------------------------------

;; ----------------------------------------------------------------------------
;; WITNESS: Extra functions.
;; ----------------------------------------------------------------------------
(defun get-witness (E)
  (let ((ans (get-witness-list E)))
    (print-debug 2 t "WITNESS IS:~%")
    (loop for i in ans do (print-debug 2 t "~A~%" i)) 
    ans))

(defun get-witness-list (E)
  (print-debug 2 t "Get-witness function called with WITNESS~%~A~%" E)
  (if (null E) nil
      (nunion (get-witness-one (car E)) (get-witness-list (cdr E)))))

(defun get-witness-one (pol)
  (if (null (POL-wit pol)) (list pol) (get-witness-list (POL-wit pol))))

(defun make-witness (pol op)
  (print-debug 5 t "Make-witness function called with POL=~A, OP=~A~%" pol op)
  (break))

(defun combine-wit (p q &optional (wit nil))
  (combine-wit* p q wit))

(defun combine-wit* (p q wit)
  (cond ((and (POL-p p) (POL-p q))
  	 (cons p (cons q wit)))
	((and (listp p) (POL-p q))
  	 (cons q p))
	(t (print-debug 5 t "Unknown case.~%") (break))))
;; ----------------------------------------------------------------------------

;; ----------------------------------------------------------------------------
;; getUpperBound: NEW INFERENCE RULE for nonlinear polynomials.....
;; ----------------------------------------------------------------------------
(defun getUpperBound (rule R S)
  (if (null rule) (return-from getUpperBound nil))
  (let ((*bound-cache* nil))
    (declare (special *bound-cache*))
    (let ((ans (getBoundRule* rule R S)))
      (if (null ans) (return-from getUpperBound nil))
      (let ((pol (POL-pol ans)) (wit (POL-wit ans)))
        ;(print-debug 3 t "BOUND OF ~A IS ~A~%" (prep:polyrepPrint (POL-pol rule)) (prep:polyrepPrint pol))
        ;(print-debug 3 t "BOUND WITNESS: ~A~%" (dpPrint wit))
        (cond ((null wit) nil)
	      ((null pol) (values 'inconsistent wit))
	      ((and (null (cdr pol)) (null (cdar pol)))
	       (if (<= (caar pol) 0) (values 'inconsistent wit) nil))
	      (t 
		;(when (every #'(lambda(x) (or (prep:parameter? (car x)) (and (cdr x) (cadr x) (caddr x)))) *bound-cache*)
		  (loop for i in *bound-cache* do 
			;(if (not (prep:parameter? (car i))) (print-debug 3 t "~A: ~A ~A~%" (car i) (cadr i) (caddr i)))
			(print-debug 3 t "~A: ~A ~A~%" (car i) (cadr i) (caddr i)))
		  (print-debug 3 t "RS is:~%R=~A~%S=~A~%" (dpPrint R) (dpPrint S))
		; when (every #'(lambda(x) (every #'(lambda(y)(prep:parameter? (car y))) (cdr x))) pol)
        	  (print-debug 3 t "BOUND OF ~A IS ~A~%" (prep:polyrepPrint (POL-pol rule)) (prep:polyrepPrint pol))
        	  (print-debug 3 t "BOUND WITNESS: ~A~%" (dpPrint wit)) ;)
		(values 'satisfiable ans)))))))

(defun getBoundRule* (rule R S)
  (let ((RS (append R S))
	(pol (chaining-dp::POL-pol rule))
	(op  (chaining-dp::POL-op  rule)))
    (multiple-value-bind (lb ub) (getLUBoundPol pol RS)
      (declare (ignore lb))
      (if ub (setf (POL-op ub) op))
      (if ub (setf (POL-wit ub) (cons rule (POL-wit ub)))) ub)))

(defun getLUBoundPol (pol RS)
  (multiple-value-bind (npol prods) (pol2factoredpol pol)
    (print-debug 3 t "Factored POL of~A IS~% ~A AND~% ~A~%" (prep:polyrepPrint pol) (prep:polyrepPrint npol) prods)
    (let ((LUs1 (mapcar #'(lambda(x) (multiple-value-bind (lb ub) (getLUBoundPP x RS) 
					 (cons (cons lb (list x)) (cons ub (list x))))) npol))
	  (LUs2 (mapcar #'(lambda(x) (multiple-value-bind (lb ub) (getLUBoundProd x RS) 
					(let* ((mono (car x)) (poly (cdr x)) 
					       (ans (polyrepMultiplyPoly (list mono) poly)))
					  (cons (cons lb ans) (cons ub ans))))) prods)))
      (print-debug 3 t "LUs computed:npol~%")
      (loop for i in LUs1 do (print-debug 3 t "LB=~A, UB=~A~%" 
			(if (caar i) (prep:polyrepPrint (POL-pol (caar i))) "unknown")
			(if (cadr i) (prep:polyrepPrint (POL-pol (cadr i))) "unknown")))
      (print-debug 3 t "LUs computed:prods~%")
      (loop for i in LUs2 do (print-debug 3 t "LB=~A, UB=~A~%" 
			(if (caar i) (prep:polyrepPrint (POL-pol (caar i))))
			(if (cadr i) (prep:polyrepPrint (POL-pol (cadr i))))))
      (values (put-together (nconc (mapcar #'car LUs1) (mapcar #'car LUs2)))
	      (put-together (nconc (mapcar #'cdr LUs1) (mapcar #'cdr LUs2)))))))

(defun put-together (PdotPPlist)
  (if (every #'null (mapcar #'car PdotPPlist)) nil
      (put-together* (mapcar #'(lambda(x)(if (car x) (car x) 
				(make-POL :pol (cdr x) :wit nil))) PdotPPlist) nil nil)))

(defun put-together* (POLlist pol wit)
  (if (null POLlist) (make-POL :pol pol :wit wit)
      (put-together* (cdr POLlist)
	(prep:polyrepAddPoly (POL-pol (car POLlist)) pol) (append (POL-wit (car POLlist)) wit))))

(defun getLUBoundPP (c0s0 RS)
  (print-debug 3 t "getLUBoundPP called: For c0s0 = ~A~%" c0s0)
  (multiple-value-bind (lb ub) (getLUBoundPP* (car c0s0) (cdr c0s0) RS)
    (print-debug 3 t "getLUBoundPP returns: LB=~A, UB=~A~%" lb ub)
    (values lb ub)))

(defun getLUBoundPP* (c0 s0 RS)
  (if (null s0) (return-from getLUBoundPP* (values (make-POL :pol (list (list c0))) (make-POL :pol (list (list c0))))))
  (let ((partition (loop for i in s0 collect
		      (multiple-value-bind (lb ub) (getLUBoundPP** (list i) RS 1 1 nil nil)
			(cond ((and lb ub (not (prep:parameter? (car i)))) (list i nil nil))
			      ((and lb (>= lb 0)) (list nil nil i))
			      ((and ub (<= ub 0)) (list nil i i))
			      (t (return -1)))))))
    (if (eq partition -1) (return-from getLUBoundPP* nil))
    (let ((goodPPs (delete-if #'null (mapcar #'car partition)))
	  (oknegPPs (delete-if #'null (mapcar #'cadr partition)))
	  (okPPs (delete-if #'null (mapcar #'caddr partition))))
      (if (null goodPPs) (return-from getLUBoundPP* nil))
      (multiple-value-bind (goodlb goodub goodw1 goodw2) (getLUBoundPP** goodPPs RS c0 c0 nil nil)
        (multiple-value-bind (coeflb coefub witlb witub)
		(if (evenp (length oknegPPs))
		    (values goodlb goodub goodw1 goodw2)
		    (values goodub goodlb goodw2 goodw1))
	  (values (make-POL :pol (polyrepMultiplyCstPoly coeflb (list (cons 1 okPPs))) :wit witlb)
		  (make-POL :pol (polyrepMultiplyCstPoly coefub (list (cons 1 okPPs))) :wit witub)))))))
		  ;; NOT EXACTLY CORRECT, param wits ignored
 
(defun getLUBoundPP** (s0 RS lb ub w1 w2)
  (if (null s0)
      (values lb ub w1 w2)
      (multiple-value-bind (lb1 ub1 w11 w21) (getLUBoundVar (caar s0) RS)
	(multiple-value-bind (lb2 ub2) (exptBound lb1 ub1 (cdar s0))
	  (multiple-value-bind (lb3 ub3 w13 w23) (multiplyBounds lb ub w1 w2 lb2 ub2 w11 w21)
	    (getLUBoundPP** (cdr s0) RS lb3 ub3 w13 w23))))))

(defun exptBound (lb ub exponent)
  (cond ((and lb (>= lb 0))			;; 0 <= lb < ?
	 (values (expt lb exponent) (if ub (expt ub exponent))))
	((and ub (<= ub 0) (evenp exponent))	;; ? < ub <= 0
	 (values (expt ub exponent) (if lb (expt lb exponent))))
	((and ub (<= ub 0) (oddp exponent))	;; ? < ub <= 0
	 (values (if lb (expt lb exponent)) (expt ub exponent)))
	((and lb ub (oddp exponent))		;; lb < 0 < ub
	 (values (expt lb exponent) (expt ub exponent)))
	((and lb ub (evenp exponent))		;; lb < 0 < ub
	 (values 0 (expt (if (> (abs lb) ub) lb ub) exponent)))
	((and ub (oddp exponent))		;; lb = nil, ub > 0
	 (values nil (expt ub exponent)))
	((and ub (evenp exponent))		;; lb = nil, ub > 0
	 (values 0 nil))
	((and lb (oddp exponent))		;; lb < 0, ub = nil
	 (values (expt lb exponent) nil))
	((and lb (evenp exponent))		;; lb < 0, ub = nil
	 (values 0 nil))
	(t					;; lb = ub = nil
	 (values nil nil))))

(defun multiplyBounds (lb1 ub1 w11 w21 lb2 ub2 w12 w22)
  (cond ((and lb1 lb2 (>= lb1 0) (>= lb2 0))			;; pos; pos; 0 <= lb1, lb2 < ?
	 (values (* lb1 lb2) (if (and ub1 ub2) (* ub1 ub2)) 
		 (append w11 w12) (if (and ub1 ub2) (append w21 w22))))
	((and ub1 ub2 (<= ub1 0) (<= ub2 0))			;; neg; neg; 0 >= ub1, ub2 > ?
	 (values (* ub1 ub2) (if (and lb1 lb2) (* lb1 lb2)) 
		 (append w21 w22) (if (and lb1 lb2) (append w11 w12))))
	((and lb1 (>= lb1 0) ub2 (<= ub2 0))			;; pos; neg
	 (values (if (and ub1 lb2) (* ub1 lb2)) (* lb1 ub2)
		 (if (and ub1 lb2) (append w21 w12)) (append w11 w22)))
	((and lb2 (>= lb2 0) ub1 (<= ub1 0))			;; neg; pos
	 (values (if (and ub2 lb1) (* ub2 lb1)) (* lb2 ub1)
		 (if (and ub2 lb1) (append w22 w11)) (append w12 w21)))
	((and lb1 (>= lb1 0))					;; pos; mixed
	 (values (if (and ub1 lb2) (* ub1 lb2)) (if (and ub1 ub2) (* ub1 ub2))
		 (if (and ub1 lb2) (append w21 w12)) (if (and ub1 ub2) (append w21 w22))))
	((and lb2 (>= lb2 0))					;; mixed; pos
	 (values (if (and ub2 lb1) (* ub2 lb1)) (if (and ub1 ub2) (* ub1 ub2))
		 (if (and ub2 lb1) (append w22 w11)) (if (and ub1 ub2) (append w21 w22))))
	((and ub1 (<= ub1 0))					;; neg; mixed
	 (values (if (and ub2 lb1) (* ub2 lb1)) (if (and lb1 lb2) (* lb1 lb2))
		 (if (and ub2 lb1) (append w22 w11)) (if (and lb1 lb2) (append w11 w12))))
	((and ub2 (<= ub2 0))					;; mixed; neg
	 (values (if (and ub1 lb2) (* ub1 lb2)) (if (and lb1 lb2) (* lb1 lb2))
		 (if (and ub1 lb2) (append w21 w12)) (if (and lb1 lb2) (append w11 w12))))
	((and lb1 lb2 ub1 ub2)					;; mixed, mixed--all known
	 (let ((min1 (* lb1 ub2)) (min2 (* lb2 ub1)) 
	       (max1 (* lb1 lb2)) (max2 (* ub1 ub2)))
	   (values (if (< min1 min2) min1 min2) (if (> max1 max2) max1 max2)
		   (if (< min1 min2) (append w11 w22) (append w12 w21)) 
		   (if (> max1 max2) (append w11 w12) (append w21 w22)))))
	(t							;; mixed, mixed--some unknown!
	 (values nil nil nil nil))))

;; Return-value: (values num num POLlist POLlist)
(defun getLUBoundVar (var RS)
  (declare (special *bound-cache*))
  (let ((ans (cdr (assoc var *bound-cache*))))
    (if ans (values (nth 0 ans) (nth 1 ans) (nth 2 ans) (nth 3 ans))
	(multiple-value-bind (lb ub w1 w2) (getLUBoundVar* var RS nil nil nil nil RS)
	  (setf *bound-cache* (acons var (list lb ub w1 w2) *bound-cache*))
	  (values lb ub w1 w2)))))

;; TYPE var: prep-var; RS: POLlist; lb, ub: number; w1, w2: POL!!
;; Return-value: (values num num POLlist POLlist)
(defun getLUBoundVar* (var RS lb ub w1 w2 *RS*)
  (if (null RS) (values lb ub (list w1) (list w2))	;; WITNESS--LIST of rules.
      (let ((p (chaining-dp::POL-pol (car RS))))
	(if (or (null p) (null (cdar p)) (not (prep:var-equal? var (caadar p))))
	    (return-from getLUBoundVar* (getLUBoundVar* var (cdr RS) lb ub w1 w2 *RS*)))
	(if (prep:interval? p)
	    (let ((val (if (cdr p) (/ (- 0 (caadr p)) (caar p)) 0)))
	      (multiple-value-bind (newlb neww1) (if (and (> (caar p) 0) (or (null lb) (> val lb))) (values val (car RS)) (values lb w1))
	      (multiple-value-bind (newub neww2) (if (and (< (caar p) 0) (or (null ub) (< val ub))) (values val (car RS)) (values ub w2))
	        (getLUBoundVar* var (cdr RS) newlb newub neww1 neww2 *RS*))))
	    (let ((newvar (prep::difference? p)))
	      (if (null newvar) (return-from getLUBoundVar* (getLUBoundVar* var (cdr RS) lb ub w1 w2 *RS*)))
	      (multiple-value-bind (lb2 ub2 w12 w22) (getLUBoundVar newvar *RS*)	;; w12, w22 are LISTs!
		(let ((coef (/ (- 0 (car (nth 1 p))) (caar p))))
		  (multiple-value-bind (lb3 ub3 w13 w23) (multiplyBounds coef coef nil nil lb2 ub2 w12 w22)
		    (multiple-value-bind (newlb newub neww1 neww2) 			;; w13, w23 are LISTs!
		      (if (> (caar p) 0) 
			  (if (or (null lb3) (and lb (>= lb lb3))) (values lb ub w1 w2) (values lb3 ub (car w13) w2))
			  (if (or (null ub3) (and ub (<= ub ub3))) (values lb ub w1 w2) (values lb ub3 w1 (car w23))))
		      (getLUBoundVar* var (cdr RS) newlb newub neww1 neww2 *RS*))))))))))
;; ----------------------------------------------------------------------------

;; ----------------------------------------------------------------------------
(defun getLUBoundProd (monodotpol RS)
  (multiple-value-bind (lb1 ub1) (getLUBoundPP* (caar monodotpol) (cdar monodotpol) RS)
    (multiple-value-bind (lb2 ub2) (getLUBoundPol (cdr monodotpol) RS)
      (if (and lb1 lb2 ub1 ub2)
	  (let* ((ub1pol (POL-pol ub1)) (ub1wit (POL-wit ub1))
		 (ub2pol (POL-pol ub2)) (ub2wit (POL-wit ub2))
		 (lb1pol (POL-pol lb1)) (lb1wit (POL-wit lb1))
		 (lb2pol (POL-pol lb2)) (lb2wit (POL-wit lb2))
		 (lb2Num (if (null lb2pol) 0 (caar lb2pol)))
		 (ub2Num (if (null ub2pol) 0 (caar ub2pol)))
		 (lb1Num (if (null lb1pol) 0 (caar lb1pol)))
		 (ub1Num (if (null ub1pol) 0 (caar ub1pol)))
		 (const? #'(lambda(x) (or (null x) (and (null (cdr x)) (null (cdar x)))))))
	    (if (and (funcall const? lb2pol) (funcall const? ub2pol))
		(cond ((or (and (> lb2Num 0) (< lb1Num ub1Num) (< ub1Num 0))
		           (and (> lb2Num 0) (>= lb1Num ub1Num) (> ub1Num 0)))
			;; 3 < 5; -5xy < -3xy (lb1.ub2 ub1.lb2)
			;; 3 < 5; 5xy < 3xy  (lb1.ub2 ub1.lb2)
		       (values (make-POL :pol (polyrepMultiplyCstPoly ub2Num  lb1pol)
				         :wit (append ub2wit lb1wit))
			       (make-POL :pol (polyrepMultiplyCstPoly lb2Num  ub1pol)
				         :wit (append lb2wit ub1wit))))
		      ((or (and (> lb2Num 0) (< lb1Num ub1Num) (> lb1Num 0))
		           (and (> lb2Num 0) (>= lb1Num ub1Num) (< lb1Num 0)))
			;; 3 < 5; 3xy < 5xy  (lb1.lb2 ub1.ub2)
			;; 3 < 5; -3xy < -5xy  (lb1.lb2 ub1.ub2)
		       (values (make-POL :pol (polyrepMultiplyCstPoly lb2Num  lb1pol)
				         :wit (append lb2wit lb1wit))
			       (make-POL :pol (polyrepMultiplyCstPoly ub2Num  ub1pol)
				         :wit (append ub2wit ub1wit))))
		      ((or (and (> lb2Num 0) (< lb1Num ub1Num))
		           (and (> lb2Num 0) (>= lb1Num ub1Num)))
			;; 3 < 5; -3xy < 5xy  (lb1.ub2 ub1.ub2)
			;; 3 < 5; 5xy < -3xy  (lb1.ub2 ub1.ub2)
		       (values (make-POL :pol (polyrepMultiplyCstPoly ub2Num  lb1pol)
				         :wit (append ub2wit lb1wit))
			       (make-POL :pol (polyrepMultiplyCstPoly ub2Num  ub1pol)
				         :wit (append ub2wit ub1wit))))
		      ((or (and (<= ub2Num 0) (< lb1Num ub1Num) (< ub1Num 0))
		           (and (<= ub2Num 0) (>= lb1Num ub1Num) (> ub1Num 0)))
			;; -5 < -3; -5xy < -3xy  (ub1.ub2 lb1.lb2)
			;; -5 < -3; 5xy < 3xy  (ub1.ub2 lb1.lb2)
		       (values (make-POL :pol (polyrepMultiplyCstPoly ub2Num  ub1pol)
				         :wit (append ub2wit ub1wit))
			       (make-POL :pol (polyrepMultiplyCstPoly lb2Num  lb1pol)
				         :wit (append lb2wit lb1wit))))
		      ((or (and (<= ub2Num 0) (< lb1Num ub1Num) (> lb1Num 0))
		           (and (<= ub2Num 0) (>= lb1Num ub1Num) (< lb1Num 0)))
			;; -5 < -3; 3xy  <  5xy  (ub1.lb2 lb1.ub2)
			;; -5 < -3; -3xy < -5xy  (ub1.lb2 lb1.ub2)
		       (values (make-POL :pol (polyrepMultiplyCstPoly lb2Num  ub1pol)
				         :wit (append lb2wit ub1wit))
			       (make-POL :pol (polyrepMultiplyCstPoly ub2Num  lb1pol)
				         :wit (append ub2wit lb1wit))))
		      ((or (and (<= ub2Num 0) (< lb1Num ub1Num))
		           (and (<= ub2Num 0) (>= lb1Num ub1Num)))
			;; -5 < -3; -3xy < 5xy  (ub1.lb2 lb1.lb2)
			;; -5 < -3; 3xy < -5xy  (ub1.lb2 lb1.lb2)
		       (values (make-POL :pol (polyrepMultiplyCstPoly lb2Num  ub1pol)
				         :wit (append lb2wit ub1wit))
			       (make-POL :pol (polyrepMultiplyCstPoly lb2Num  lb1pol)
				         :wit (append lb2wit lb1wit))))
		      ((or (and (< lb1Num ub1Num) (< ub1Num 0))
		           (and (>= lb1Num ub1Num) (> ub1Num 0)))
			;; -3 < 5; -5xy < -3xy  (lb1.ub2 lb1.lb2)
			;; -3 < 5; 5xy < 3xy	(lb1.ub2 lb1.lb2)
		       (values (make-POL :pol (polyrepMultiplyCstPoly ub2Num  lb1pol)
				         :wit (append ub2wit lb1wit))
			       (make-POL :pol (polyrepMultiplyCstPoly lb2Num  lb1pol)
				         :wit (append lb2wit lb1wit))))
		      ((or (and (< lb1Num ub1Num) (> lb1Num 0))
		           (and (>= lb1Num ub1Num) (< lb1Num 0)))
			;; -3 < 5; 3xy < 5xy  (ub1.lb2 ub1.ub2)
			;; -3 < 5; -3xy < -5xy  (ub1.lb2 ub1.ub2)
		       (values (make-POL :pol (polyrepMultiplyCstPoly lb2Num  ub1pol)
				         :wit (append lb2wit ub1wit))
			       (make-POL :pol (polyrepMultiplyCstPoly ub2Num  ub1pol)
				         :wit (append ub2wit ub1wit))))
		      ((and (< lb1Num ub1Num)) 
			;; -3 < 5; -3xy < 5xy  (min(lb1.ub2,ub1.lb2), max(ub1.ub2,lb1.lb2))
		       (values
		         (if (< (* lb1Num ub2Num) (* ub1Num lb2Num))
			     (make-POL :pol (polyrepMultiplyCstPoly ub2Num lb1pol)
				       :wit (append ub2wit lb1wit))
			     (make-POL :pol (polyrepMultiplyCstPoly lb2Num ub1pol)
				       :wit (append lb2wit ub1wit)))
		         (if (> (* ub1Num ub2Num) (* lb1Num lb2Num))
			     (make-POL :pol (polyrepMultiplyCstPoly ub2Num ub1pol)
				       :wit (append ub2wit ub1wit))
			     (make-POL :pol (polyrepMultiplyCstPoly lb2Num lb1pol)
				       :wit (append lb2wit lb1wit)))))
		      ((and (>= lb1Num ub1Num))
			;; -3 < 5; 5xy < -3xy  (max(lb1.ub2,ub1.lb2), min(ub1.ub2,lb1.lb2))
		       (values
		         (if (> (* lb1Num ub2Num) (* ub1Num lb2Num))
			     (make-POL :pol (polyrepMultiplyCstPoly ub2Num lb1pol)
				       :wit (append ub2wit lb1wit))
			     (make-POL :pol (polyrepMultiplyCstPoly lb2Num ub1pol)
				       :wit (append lb2wit ub1wit)))
		         (if (< (* ub1Num ub2Num) (* lb1Num lb2Num))
			     (make-POL :pol (polyrepMultiplyCstPoly ub2Num ub1pol)
				       :wit (append ub2wit ub1wit))
			     (make-POL :pol (polyrepMultiplyCstPoly lb2Num lb1pol)
				       :wit (append lb2wit lb1wit)))))
		      (t (format t "UNREACHABLE CODE. CHECK.~%") (break)))
		nil))
	  nil))))
  
;; input: pol; output: pol as (values good-part (list-of-(cons mono . pol)))
(defun pol2factoredpol (pol)
  (let* ((posmonos (loop for i in pol if (> (car i) 0) collect i))
	 (negmonos (loop for i in pol if (< (car i) 0) collect i))
	 (posvars (delete-if #'prep:parameter? (prep:allVarsIn posmonos)))
	 (negvars (delete-if #'prep:parameter? (prep:allVarsIn negmonos))))
    (if (null (intersection posvars negvars)) (return-from pol2factoredpol (values pol nil)))
    (multiple-value-bind (goodposmonos badposmonos) (splitOnVars posmonos negvars)
    (multiple-value-bind (goodnegmonos badnegmonos) (splitOnVars negmonos posvars)
	(mergebadmonos badposmonos badnegmonos (append goodposmonos goodnegmonos) nil)))))

(defun mergebadmonos (posmonos negmonos &optional (goodres nil) (badres nil))
  (if (null posmonos) (values (append negmonos goodres) badres)
      (let* ((this (car posmonos))
	     (match (loop for i in negmonos 
			if (and (or (every #'(lambda(x) (member x (cdr this) :test #'equal)) (cdr i))
				    (every #'(lambda(x) (member x (cdr i) :test #'equal)) (cdr this)))
				(eq (abs (- (length (cdr i)) (length (cdr this)))) 1)) return i))
	     (restneg (delete match negmonos))
	     (newmono (if match (factorAndAddMonos this match))))
	(mergebadmonos (cdr posmonos) restneg (if match goodres (cons this goodres))
		(if match (cons newmono badres) badres)))))

(defun factorAndAddMonos (c0s0 d0t0)
  (let* ((c0 (car c0s0))
	 (d0 (car d0t0))
	 (flag (< (length c0s0) (length d0t0)))
	 (new (if flag  (set-difference (cdr d0t0) (cdr c0s0) :test #'equal)
			(set-difference (cdr c0s0) (cdr d0t0) :test #'equal))))
    (if flag
	(cons c0s0 (list (cons (/ d0 c0) new) (list 1)))
	(cons d0t0 (list (cons (/ c0 d0) new) (list 1))))))

(defun splitOnVars (monos vars)
  (let ((badmonos (loop for i in monos if
			(let ((thisvars (loop for j in (cdr i) if (not (prep:parameter? (car j))) collect (car j))))
	  		  (some #'(lambda(x)(member x vars)) thisvars)) collect i)))
    (values (set-difference monos badmonos) badmonos)))
;; ----------------------------------------------------------------------------

;; Is this polynomial linear?
(defun linearPoly? (poly)
  (if (null poly) t
      (if (linearPP? (cdar poly)) (linearPoly? (cdr poly)) nil)))

(defun linearPP? (mu)
  (cond ((null mu) t)
	((and (null (cdr mu)) (eq (cdar mu) 1)) t)
	(t nil)))

(defun dpPrint (ERS)
  (mapcar #'(lambda(x) (prep:polyrepPrint (POL-pol x))) ERS))
