;; --------------------------------------------------------------------
;; HybridSAL
;; Copyright (C) 2006, SRI International.  All Rights Reserved.
;; 
;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License
;; as published by the Free Software Foundation; either version 2
;; of the License, or (at your option) any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
;; --------------------------------------------------------------------

;;;;;;;;;;;;;;;;;;;;;;;;;;* -*- Mode: Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; vim: syntax=lisp
;; sal-extension.lisp --
;; Author          : Ashish Tiwari
;; Created On      : Fri Oct 18, 2002
;; Last Modified By: Ashish Tiwari
;; Last Modified On: Fri Oct 18, 2002
;; Update Count    : 0
;; Status          : Unknown, use with caution
;;
;; HISTORY : 
;; 10.19.02: SAL extension routines for abstraction.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "sal.hsal")

;(export '(mk-syntactic-bool* mk-sal-equation mk-sal-application mk-sal-nameexpr module2moddecl))	;; id used by hsal-abs-sat.lisp

;(export '(synchronous-composition guard expression))

(defvar *lcache* nil)	;; condition-value-pair list
(defstruct CV tt ff val)	;; conditional-value
;; ============================================================================
;; Input: list of simple-defns and gcs. Return true if this is piecewise-cont
;; hybrid system.
;; ============================================================================
(defmethod piecewise-continuous? ((module module))
  (piecewise-continuous? (definitions-or-commands 
	(find-if #'trans-decl? (declarations module)))))

(defmethod piecewise-continuous? ((decls list))
  (and (find-if #'simple-definition? decls)
       (let ((gc (find-if #'guarded-command? decls)))
	 (and gc (continuous-tran? gc)))))

(defun get-rhs-exprs (decls)
  (loop for i in decls collect 
	(if (simple-definition? i) (rhs-definition i)
	    (expression (guard i)))))

(defun get-conditions (expr)
  (if (or (conditional? expr) (chained-conditional? expr))
      (cons (car (exprs (argument expr)))
	    (nconc (get-conditions (cadr (exprs (argument expr))))
		   (get-conditions (caddr (exprs (argument expr))))))
      nil))

;; return: ((var . ( (c1.v1) (c2.v2) ... (cn.vn) )) (var . ( ... )) (var . (...)))
;; ci: (trueC . falseC), each is a listof conditions.
(defun get-condition-value-pair-list (decls symtab)
  (declare (special *lcache*))
  (let* ((ans0 (cdr (assoc decls *lcache* :test #'equal)))
    	 (ans1 (if ans0 ans0 (get-condition-value-pair-list* decls symtab nil))))
    (if (not ans0) (setf *lcache* (acons decls ans1 *lcache*)))
    ;; (format t "Get-condition-value-pair%sal.hsal::*lcache* and decls present?~%")
    ;; (break)
    ans1))

(defun get-condition-value-pair-list* (decls symtab res)
  (cond ((null decls) res)
	((guarded-command? (car decls))
	 (get-condition-value-pair-list* (cdr decls) symtab res))
	((simple-definition? (car decls))
	 (let* ((sd (car decls))
		(lhs (lhs sd))
		(rhs0 (rhs-definition sd)))
	   (get-condition-value-pair-list* (cdr decls) symtab 
		(acons lhs (get-cvp-list rhs0 symtab) res))))
	(t (sal-error t "Unreachable code reached.~%"))))

(defun get-cvp-list (expr symtab)
  (get-cvp-list* (list (make-CV :tt nil :ff nil :val expr)) nil symtab))

(defun get-cvp-list* (nodes res symtab)
  (if (null nodes) res
      (let ((trueC (CV-tt (car nodes)))
	    (falseC (CV-ff (car nodes)))
	    (expr (CV-val (car nodes))))
	(cond ((sal.hsal.abs:ITE-p expr)
	       (let* ((c1 (sal.hsal.abs:ITE-c expr))
		      (v1 (sal.hsal.abs:ITE-v1 expr))
		      (v2 (sal.hsal.abs:ITE-v2 expr))
		      (new1 (make-CV :tt (cons c1 trueC) :ff falseC :val v1))
		      (new2 (make-CV :tt trueC :ff (cons c1 falseC) :val v2)))
	         (get-cvp-list* (cons new1 (cons new2 (cdr nodes))) res symtab)))
	      ((sal.hsal.abs:RAF-p expr)	;; real value: terminate.
	       (get-cvp-list* (cdr nodes) (cons (car nodes) res) symtab))
	      (t				;; recurse, SAL-ITE2My-ITE
	       (let* ((expr1 (cdr (assoc expr symtab))))
		 (assert expr1)
		 (setf (CV-val (car nodes)) expr1)
		 (get-cvp-list* nodes res symtab)))))))
;; ============================================================================
  
;; ============================================================================
(defmethod hybrid-automaton? ((module module))
  (hybrid-automaton? (definitions-or-commands 
	(find-if #'trans-decl? (declarations module)))))

(defmethod hybrid-automaton? ((cmds list))
  (and (every #'guarded-command? cmds)
       (some #'continuous-tran? cmds)))
 
(defun continuous-tran? (tran)
  (and (guarded-command? tran)
       (dummy-variable? (lhs (car (assignments tran))))))

(defun get-guards (cmds)
  (loop for i in cmds if (guarded-command? i) collect (expression (guard i))))

(defmethod dummy-variable? ((lhs next-operator))
  (dummy-variable? (id (name lhs))))

(defmethod dummy-variable? ((name name-expr))
  (dummy-variable? (id name)))

(defmethod dummy-variable? ((symb symbol))
  (let* ((str (symbol-name symb)) (len (length str)))
    (and (> len 3) (string= str "dot" :start1 (- len 3)))))

(defun sal-parse-list (ctxt module pols)
  (if (null pols) nil
      (sal-parse-list* ctxt module pols)))

(defun sal-parse-list* (ctxt modul pols &optional (res '()))
  (if (null pols) (nreverse res)
      (let* ((npol (sal-parse-term (car pols)))) ;; :nt is 'expression
	     
        (typecheck npol sal:'REAL)
	(sal-parse-list* ctxt modul (cdr pols) (cons npol res)))))

(defun get-flow (module assgns &optional (res nil))
  (if (null assgns) res
      (let* ((assgn1 (car assgns))
	     (lhsvar (lhs assgn1))	;; assgn1 is simple-definition
	     (newvar (dummy-var2real-var lhsvar))
	     (rhs (rhs-definition assgn1));; varname is name-expr
	     (newrhs (pvs2polyrep:pvs2polyrep rhs)))
	(get-flow module (cdr assgns) (acons newvar newrhs res)))))

(defun dummy-var2real-var (lhsvar)
  (let* ((var (if (name-expr? lhsvar) lhsvar (name lhsvar)))	;; lhsvar is next-operator
    	 (rvar (find var (prep:get-order) :test #'(lambda (x y)
                (let* ((str1 (symbol-name (sal:id x)))
                       (str2 (symbol-name (sal:id y)))
                       (len1 (length str1))
                       (len2 (length str2)))
                  (and (eq (+ len2 3) len1) (string= str1 str2
                         :end1 (length str2))))))))
    (if rvar (return-from dummy-var2real-var rvar))
    (format t "Variable ~A not found in database.~%" lhsvar)
    (break)))
;; ============================================================================

;; ============================================================================
;; New mk-sal-functions
;; ============================================================================
;; Return sal-conjunction of all elements in input list
(defun mk-syntactic-bool (elmts flag)
  (let* ((cst (if (eq flag sal:'AND) sal:'TRUE sal:'FALSE))		;; id
  	 (inv (if (eq flag sal:'AND) sal:'FALSE sal:'TRUE))		;; inverse
	 (fn #'(lambda (x) (and (name-expr? x) (eq (id x) cst))))
	 (fn1 #'(lambda (x y) (and (name-expr? y) (eq (id y) x))))
  	 (elmts1 (remove-if fn elmts)))
    (if (member inv elmts1 :test fn1) (mk-sal-nameexpr nil inv) 
        (mk-syntactic-bool** elmts1 cst flag))))

(defun mk-syntactic-bool* (elmts op)
  (mk-syntactic-bool** elmts (if (eq op sal:'AND) sal:'TRUE sal:'FALSE) op))

(defun mk-syntactic-bool** (elmts id op)
  (cond ((null elmts) (mk-sal-nameexpr nil id))
	((null (cdr elmts)) (car elmts))
	(t
	 (mk-sal-bool (car elmts) (mk-syntactic-bool** (cdr elmts) id op) op))))

(defun module2moddecl (amod cmod)
  (let* ((mid (id (mod-name (modref cmod))))
	 (amodname (mk-sal-modulename nil mid))
	 (avardecls (mk-sal-vardecls nil))) 	;; HERE: remove one nil
    (mk-sal-moduledeclaration nil amodname avardecls amod)))
;; ============================================================================

;; ============================================================================
;; RESET between different abstraction runs.
(defun reset ()
  (declare (special hsal-abs-real::*cache* hsal-abs-real::*neg-cache* hsal-abs-real::*p2absp-cache* sal.hsal.pl::*p2cp-cache*))
  (setf hsal-abs-real::*cache* nil)		;; condition --> a1234 (= > >= <> RAFs)
  (setf hsal-abs-real::*neg-cache* nil)		;; similarly for neg(condition)
  (setf hsal-abs-real::*p2absp-cache* nil)	;; pol --> conditions under which p>=<0
  (setf sal.hsal::*lcache* nil)			;; (var-(cond-value-list))-list
  (setf sal.hsal.pl::*p2cp-cache* nil))		;; pol --> (cond.p1)-list
;; ============================================================================

