;; --------------------------------------------------------------------
;; HybridSAL
;; Copyright (C) 2006, SRI International.  All Rights Reserved.
;; 
;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License
;; as published by the Free Software Foundation; either version 2
;; of the License, or (at your option) any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
;; --------------------------------------------------------------------

;;;;;;;;;;;;;;;;;;;;;;;;;;* -*- Mode: Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; vim: syntax=lisp
;; hsal-compose.lisp --
;; Author          : Ashish Tiwari
;; Created On      : Wed Sep 11, 2002
;; Last Modified By: Ashish Tiwari
;; Last Modified On: Wed Sep 11, 2002
;; Update Count    : 0
;; Status          : Unknown, use with caution
;;
;; HISTORY : 
;; 09.11.02: Compositional Abstraction
;; 09.26.02: Local vars in ABS are declared GLOBAL (because of model-checking)
;; 10.19.02: packages, seed-polynomial not everything from database,
;;	database has invariant formula too.
;; 10.21.02: dummy-variable?: INPUT is name-expr? and not a symbol now??!!
;; 12.05.02: def-decl added to get-seed-pols.
;; 12.20.02: Trying to handle DEF-DECL more completely
;; 05.28.03: Bug in calling prep:set-variable. Called after saturate-invariant-set!
;; 06.09.03: In saturate-invariant-set: setting POL-wit to NIL.
;; 06.19.03: Bug in getExactlyRelevantPols1 corrected.
;; 06.19.03: DTS transitions abstract-hybrid-trans called!
;; 06.21.03: INV-calls nil for DTSs. Fixed. Check for nullness.
;; 06.22.03: twoModes2OneMode function added.
;; 06.27.03: abstract-property-on-pols1B can handle booleans. seealso pvs-ext.lisp
;; 07.08.03: bug in consistent-mode : nconc called inside LOOP: pulled outside!!
;; 09.02.03: *namap* (& *feasL*) introduced to keep track of negs of amap pols.
;; 09.25.03: set-union: test made stronger.
;; 	(o.w. duplicates ACROSS modules get diff names!)
;; 09.25.03: get-abstraction-mapping: chk? added. (This is overkill now?)
;; 09.25.03: abstract-on-pols: *pols* set for DTS too.
;; 09.25.03: get-abstraction-mapping: nnormalize->normalize
;; 09.25.03: get-abstraction-mapping: because of that p1->p0 (diff ptrs now)
;; 09.26.03: sal.hsal.pl:abstract-piecewise-system was ignoring *in-amap*!!!!
;; 09.26.03: test added when defining in-amap.
;; 01.28.04: infer-sign introduced; changed abstract-initialization
;; 04.06.04: :property 'all means abstract all properties
;; 05.29.07: partition-on-types real-vars were pvs-typechecked b4
;; 03.05.09: cabstract* on synchronous-composition : removed list 
;; 11.09.10: abstract-cont-tran1: (guard AND guard') to ensure next-state in state-inv
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; GLOBAL variables:
;; *context* : pointer to the current context
;; *ilog-list*: ( (i1,l1,o1,g1) (i2,l2,o2,g2) (in,ln,on,gn) ) 
;; *dbs*: ( db1 db2 ... dbn ), dbi: (decl.(fmla.nfmla)-list)-list
;; *maxdegree*: degree of derivatives included in P <= this number.
;; *optimize*: Needed by decide for optimizations in redundancy.
;; *symtab*: { sal-expr } --> {BAF, RAF, DNF}. *dbs*:list of *symtab*s
;; *symtab1*: { guard-expr } --> {a1, a2, a3, a4}.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package :sal)

;; REQUIRES: pvs2prep:pvs2polyrep, sal:pvsexpr2RAFstrings, prep:*,

;; ============================================================================
;; abstract: (compositional) abstraction. THIS IS A WRAPPER IN SAL package.
;; INPUT: context-id, module-id
;; OPTIONAL INPUT: property, seed polynomials, depth of saturation,
;;	polynomials-from-guards-included?, maxdegree of pols to be considered.
;; OUTPUT: abstracted module
;; ============================================================================
(defmethod abstract ((filename string) (mid symbol)
		&key (property nil) (pols nil) (depth 2) (more? t)
		     (optimize? t) (maxdegree 2) (dlevel 4))
  (let* ((*context* (sal:sal-parse filename))
	 (*context* (sal:typecheck* *context*))
	 (*current-sal-moddecl* 
		(find mid (context-body *context*) :key #'id))
	 (*current-sal-module* (module *current-sal-moddecl*))
	 (c *context*))
    (declare (special *context*)) ;; ADDED THIS LATER !!
    (declare (special *current-sal-module*))
    (sal.hsal::reset)			;; RESET ALL caches
    (abstract c *current-sal-moddecl* :property property :pols pols :depth depth :more? more? :optimize? optimize? :maxdegree maxdegree :dlevel dlevel)))

(defmethod abstract ((c context) (mid symbol)
		&key (property nil) (pols nil) (depth 2) (more? t)
		     (optimize? t) (maxdegree 2) (dlevel 4))
  (let* ((*context* c)
	 (*current-sal-moddecl* 
		(find mid (context-body *context*) :key #'id))
	 (*current-sal-module* (module *current-sal-moddecl*)))
    (declare (special *context*))	;; I ADDED THIS HERE LATER !!
    (declare (special *current-sal-module*))
    (sal.hsal::reset)			;; RESET ALL caches
    (abstract c *current-sal-moddecl* :property property :pols pols :depth depth :more? more? :optimize? optimize? :maxdegree maxdegree :dlevel dlevel)))

(defmethod abstract ((c context) (mdecl module-declaration)
		&key (property nil) (pols nil) (depth 2) (more? t)
		     (optimize? t) (maxdegree 2) (dlevel 4))
  (let* ((prop (hsal-abs:get-property c property))
  	 (ppols (if prop (hsal-abs:polynomials-in-property prop)))
	 (allpols (nconc ppols pols))
	 (pols1 (sal.hsal:sal-parse-list c (module mdecl) allpols)))	;;do in SAL::
    (hsal-abs:abstract* c mdecl prop pols1 depth more? optimize? maxdegree dlevel)))

(in-package "hsal-abs")

(defmacro print-debug (level &rest others)
  `(if (and hsal-abs:*dlevel* (> ,level *dlevel*))
       (funcall #'format ,@others)))

;; ============================================================================
;; Data-structures:
;; ============================================================================
;; (defstruct BAF var val sgn)	;; Boolean atomic formula
;; (defstruct RAF pol op)		;; Real atomic formula
;; (defstruct DNF products)	;; Disjunctive Normal Form
;; (defstruct ITE c v1 v2)		;; If c THEN v1 ELSE v2
;; (defstruct RES amod adecls amap order parameters E0 R0 S0);; Result of abstraction
;; abstract mod-decls, other abstract decls, rest is self-explanatory
;; ============================================================================

;; Abstract the property and Write Files.
(defun abstract* (c mdecl property pols depth more? optimize? maxdegree dlevel)
  (let ((*dlevel* dlevel)) (declare (special *dlevel*))
    (chaining-dp:set-optimize-flag optimize?)
    (hsal-abs-sat:set-maxdegree maxdegree)
    (prep:set-var-equal? #'(lambda(x y)(eq (sal:id x) (sal:id y))))
    (let* ((a-info (abstract-context c mdecl pols depth more?))
	   (amap (RES-amap a-info))
	   (order (RES-order a-info))
	   (params (RES-parameters a-info))
	   (actxt (RES-amod a-info))
	   (oldcbody (context-body actxt))
           (abspro (if property (abstract-property-on-pols property amap order params))))
	   ;; (cid (id c))
      (nconc oldcbody abspro)
      (sal.hsal:mk-sal-abstract-file actxt amap)
      ;; (sal.hsal:mk-lisp-aux-file cid order params amap 
		;; (RES-E0 a-info) (RES-R0 a-info) (RES-S0 a-info))
      ;; (sal.hsal:mk-scm-aux-file cid amap)
      ;; actxt
      "Abstraction Created.")))

;; At the context level.
(defun abstract-context (c mdecl pols depth more?)
  (let* ((amod-info (cabstract* mdecl pols depth more?))
	 (amods (RES-amod amod-info))
	 (typedecl (mk-sal-typedeclaration nil 
			     (mk-sal-nameexpr nil 'SIGN)
			     (mk-sal-scalartype nil 
				(mk-sal-scalarelement nil 'pos)
				(mk-sal-scalarelement nil 'neg)
				(mk-sal-scalarelement nil 'zero))))
	 (ctxtstr (concatenate 'string (symbol-name (id c)) "ABS"))
	 (ctxtid  (make-symbol ctxtstr))
	 (adecls1 (nconc (RES-adecls amod-info) amods))
	 (ctxtbdy (apply #'mk-sal-contextbody nil (cons typedecl adecls1)))
	 (params  (mk-sal-parameters nil))
	 (absctxt (mk-sal-context nil ctxtid params ctxtbdy)))
      (setf (RES-amod amod-info) absctxt)
      amod-info))

(defmethod cabstract* ((mdecl module-declaration) pols depth more?)
  (let* ((amod-info (cabstract* (module mdecl) pols depth more?))
	 (amod-list (RES-amod amod-info))
	 (abs-mod (sal.hsal:module2moddecl (car amod-list) (module mdecl))))
    (setf (RES-amod amod-info) (nconc (cdr amod-list) (list abs-mod)))
    amod-info))
    ;(values ctxtstr absctxt *amap* *amap-ext* *order* *parameters* *E0* *R0*)

;; ilog-list: (Input,Local,Output,Global)-vars for each module (Bool & Real).
;; gorder: global order on all variables.
;; dbs: databases, one for each basemodule
;; base-pols: seed polynomials, one for each base-module
;; transfers: all polynomials that are to be transferred
;; *amap-ext*: amap for TRANSFER polynomials....all mods use same name...
;; *assgns*: list of length 6: ASSV{PN}, ASS{CD}123, ASSV{CD}123
(defmethod cabstract* ((module synchronous-composition) pols depth more?)
  (let* ((mod-list (get-all-components* module))
         (amod-info (abstract-list* mod-list pols depth more?))
	 ;(sysmod (list (mk-sal-modulecomposition* 'syn mod-list)))
	 (sysmod (mk-sal-modulecomposition* 'syn mod-list))
	 (amod-list (RES-amod amod-info))
	 (all-mod-decls (loop for i in amod-list as j in mod-list 
				collect (sal.hsal:module2moddecl i j))))
    (setf (RES-amod amod-info) (cons sysmod all-mod-decls))
    amod-info))

(defmethod cabstract* ((module asynchronous-composition) pols depth more?)
  (let* ((mod-list (get-all-components* module))
         (amod-info (abstract-list* mod-list pols depth more?))
	 ;(sysmod (list (mk-sal-modulecomposition* 'asyn mod-list)))
	 (sysmod (mk-sal-modulecomposition* 'asyn mod-list))
	 (amod-list (RES-amod amod-info))
	 (all-mod-decls (loop for i in amod-list as j in mod-list 
				collect (sal.hsal:module2moddecl i j))))
    (setf (RES-amod amod-info) (cons sysmod all-mod-decls))
    amod-info))

(defmethod cabstract* ((module base-module) pols depth more?)
  (let* ((mod-list (list module))
	 (amod-info (abstract-list* mod-list pols depth more?)))
    amod-info))

(defun abstract-list* (mod-list pols d more?)
  (declare (special *dlevel*))
  (print-debug 8 t "Component modules: ~A~%" 
    (mapcar #'(lambda (x) (id (mod-name (modref x)))) mod-list))
  (print-debug 8 t "Partitioning variables....~%")
  (let* ((ilog-list (loop for i in mod-list collect (get-ILOG-list i)))
	 (param-list (loop for i in mod-list collect (get-parameters i)))
	 (gorder (get-all-variables ilog-list)) ;; required for build-database/pvs2polyrep
	 (gparams (loop for i in param-list append i))
	 (*gensym-counter* 0))
    (declare (special *gensym-counter*))
    (print-debug 8 t "Partitioning variables...done.~%")
    (prep:set-variables gorder)
    (prep:set-parameters gparams)
    (print-ilog-list mod-list ilog-list)
    (print-debug 8 t "Constructing database...~%")
    (let* ((db0 (hsal-abs-db:seed-database pols))	;; all databases are seeded with this
	 (dbs (loop for i in mod-list collect (hsal-abs-db:build-database i db0)))
	 (seed-pols (loop for i in mod-list 
			as j in ilog-list
			as k in dbs collect (get-seed-pols i j k pols)))
	 (base-pols (mapcar #'cdr seed-pols))
	 (transfers (set-union (mapcan #'car seed-pols) nil))	;; uniquify
	 (*amap-ext* (get-abstraction-mapping transfers #'car nil t))
	 (*assgns* (list nil nil nil nil nil nil)))
    (declare (special *amap-ext* *assgns*))
    (print-debug 8 t "Constructing database done.~%")
    (print-debug 8 t "Transfer polynomials and seed polynomials computed.~%")
    (print-seed-pols mod-list seed-pols transfers *amap-ext*)
    (let* ((amod-info-list (loop for i in mod-list 
	   		as j in ilog-list
	   		as k in dbs
	   		as l in base-pols as m in param-list collect 
	  		(abstract-basemodule i j m k l transfers gorder d more?)))
	   (amod-list (loop for i in amod-info-list collect (RES-amod i)))
	   (gamap (loop for i in amod-info-list nconc (RES-amap i)))
	   (gparams (loop for i in amod-info-list nconc (RES-parameters i)))
	   (gE0 (loop for i in amod-info-list nconc (RES-E0 i)))
	   (gR0 (loop for i in amod-info-list nconc (RES-R0 i)))
	   (gS0 (loop for i in amod-info-list nconc (RES-S0 i)))
	   (INV-calls (loop for i in amod-info-list if (RES-adecls i) collect it))
	   (fundecs (sal.hsal:get-fundec *assgns*)))
      (if (set-difference *amap-ext* gamap)
	  (format t "ERROR: Detected impure polynomial.
	  	There may be a polynomial in property (or the inputs) 
		which is not pure. ~%~A~%" (set-difference *amap-ext* gamap)))
      ;(break)
      (make-RES :amod amod-list :adecls (nconc fundecs INV-calls) :order gorder 
		:amap gamap :parameters gparams :E0 gE0 :R0 gR0 :S0 gS0)))))

(defun mk-sal-modulecomposition* (flag mod-list &optional (res nil))
  (cond ((and (null res) (or (null mod-list) (null (cdr mod-list))))
         (sal-error t "Composing zero/one modules?~%"))
	((null mod-list) res)
      	(t
	 (let* ((mod1 (mk-sal-moduleinstance nil (mk-sal-modulename nil 
			(id (mod-name (modref (car mod-list))))) nil))
	     	(mod2 (if res res
		       (mk-sal-moduleinstance nil (mk-sal-modulename nil 
			(id (mod-name (modref (cadr mod-list))))) nil)))
	     	(nres (if (eq flag 'syn) 
			  (mk-sal-synchronouscomposition nil mod1 mod2)
			  (mk-sal-asynchronouscomposition nil mod1 mod2)))
	     	(nmodlist (if res (cdr mod-list) (cddr mod-list))))
	   (mk-sal-modulecomposition* flag nmodlist nres)))))

;; ============================================================================
;; Abstract this basemodule, using the seed-pols+relevant others
;; ============================================================================
(defun abstract-basemodule (module ilog params db seed-pols others order depth more?)
  (declare (special *dlevel*))
  (let* ((outputvars (nth 6 ilog))
	 (inputvars (nth 4 ilog))
	 (extrapols (getExactlyRelevantPols1 others outputvars))
	 (inpols (getExactlyRelevantPols1 others inputvars))
	 (basepols (set-union (if more? seed-pols) extrapols))
    	 (err-pols (getWeaklyRelevantPols1 basepols inputvars)))
    (print-debug 4 t "Module ~A: inpols = ~A~%" (mod-name (modref module)) inpols)
    (if err-pols (progn (print-debug 8 t "ERROR: Cannot abstract compositionally. The module ~A contains polynomials that contain INPUT and LOCAL/OUTPUT variables SIMULTANEOUSLY.~%~A~%" (id (mod-name (modref module))) err-pols) (break)))
    (abstract-on-pols module ilog params db basepols inpols order depth)))

;; seed-pols: P_0. 
;; inpols: input pols, not in P_0, but used for constructing abstraction.
(defun abstract-on-pols (module ilog lparams db seed-pols inpols order depth)
  (declare (special *amap-ext* sal::*context* *dlevel*))
  (print-debug 8 t "------------------------------~%Abstracting MODULE ~A~%------------------------------~%" (mod-name (modref module)))
  (print-debug 8 t "Saturating the invariant set...~%")
  (let* ((inputs (nth 4 ilog))
	 (locals (nth 5 ilog))
	 (outputs (nth 6 ilog))
	 (globals (nth 7 ilog))
	 (all (append inputs locals outputs globals))
	 (lorder (loop for i in order if (member (id i) all :key #'id) collect i))
	 (*input* inputs)
	 (E0R0S0 (saturate-invariant-set module db lorder lparams))
	 (*E0* (car E0R0S0))
	 (*R0* (cadr E0R0S0))
	 (*S0* (caddr E0R0S0))
	 (*symtab* db)
	 (*current-sal-module* module))
    (declare (special *input*))
    (declare (special *E0* *R0* *S0*))
    (declare (special *symtab*))
    (declare (special *current-sal-module*))
    (print-debug 8 t "Saturating the invariant set....Done.~%")
    (print-debug 8 t "Generating polynomials for abstraction....~%")
    ;(prep:set-variables lorder)	Set in saturate-invariant-set
    ;(prep:set-parameters lparams)	Set in saturate-invariant-set
    (multiple-value-bind (*pols* *rest*)
	(cond ((sal.hsal:piecewise-continuous? module)
    	       (sal.hsal.pc.sat:saturate-pols-new db module seed-pols depth *E0* *R0* *S0*))
	      ((sal.hsal:hybrid-automaton? module)
    	       (hsal-abs-sat:saturate-pols-new sal::*context* module seed-pols depth *E0* *R0* *S0*))
	      (t (values (loop for i in seed-pols collect (list i 0 i)) nil)))
      (declare (special *pols* *rest*))
      (print-debug 8 t "Generating polynomials for abstraction...Done.~%")
      (print-all-polynomials *pols* *rest*)
      (print-debug 8 t "Computing the FEASIBILITY information. This may take a while.....~%")
      (let* ((*amap* (get-abstraction-mapping *pols* #'caddar *amap-ext* t))
	     (*namap* (loop for i in *amap* collect (cons (cdr i) (prep:polyrepNegativePoly (cdr i)))))
	     (test #'(lambda(x y)(prep:polyrepConstMultiple? x y)))
	     (*in-amap* (loop for i in inpols collect (rassoc i *amap-ext* :test test))))
        (declare (special *amap* *namap* *in-amap*))
	(if (member nil *in-amap*) (break))
	(multiple-value-bind (*INV-call* INV-decl *feasL*)
	     (get-feasible-states (mapcar #'cdr *amap*) *namap* *E0* *R0* *S0* *amap*)
	  (declare (special *INV-call* *feasL*))
	  ;; (format t "Feasible states~%") (unparse INV-decl)
	  (print-debug 8 t "Computing the FEASIBILITY information....Done.~%")
          (print-debug 8 t "Starting to abstract the dynamics w.r.t the generated polynomials....~%")
          (let* ((amod (abstract-on-pols* module nil)))
  	    (print-debug 8 t "Abstracting MODULE ~A Done.~%" (mod-name (modref module)))
	    (make-RES :amod amod :amap *amap* :order lorder :adecls INV-decl
		      :parameters lparams :E0 *E0* :R0 *R0* :S0 *S0*)))))))
  
;; *amap*: local polynomials + output polynomials (local only)
;; *amap-ext*: all output + input polynomials (global)
;; *in-amap*: input polynomials (local only)
(defmethod abstract-on-pols* ((module base-module) db)
  (declare (special *amap* *in-amap* *amap-ext*))
  (let* ((decls (declarations module))		;; get other decls
	 (newdecls (get-abstract-var-decls *amap* *in-amap* *amap-ext*))
         (adecls (abstract-on-pols-list decls newdecls db))) ;; all together
      (apply #'mk-sal-basemodule nil adecls)))

(defmethod abstract-on-pols* ((module module) db)
  (declare (special *dlevel*))
  (declare (ignore db))
  (print-debug 10 t "PANIC: Can only handle base-modules yet~%")
  module)

(defun abstract-on-pols-list (decls newdecls &optional (db nil))
  (if (null decls) (nreverse newdecls)
      (multiple-value-bind (adecl1 db1) (abstract-on-pols* (car decls) db)
	(cond  ((null adecl1) 
		(abstract-on-pols-list (cdr decls) newdecls db1))
	       ((listp adecl1)
		(abstract-on-pols-list (cdr decls) (append adecl1 newdecls) db1))
	       (t
		(abstract-on-pols-list (cdr decls) (cons adecl1 newdecls) db1))))))

(defmethod abstract-on-pols* ((vdecl var-decl) db)
  (let* ((var-type (sal::declared-type vdecl)))
    (if (sal:tc-eq vdecl 'sal::REAL) 
	(values nil db)
	(let ((vardecl (mk-sal-vardecl nil 
			(mk-sal-nameexpr nil (id vdecl)) var-type)))
	  (cond ((sal:global-decl? vdecl)
		 (values (mk-sal-globaldecl nil vardecl) db))
		((sal:local-decl? vdecl)
		 (values (mk-sal-localdecl nil vardecl) db))
		((sal:input-decl? vdecl)
		 (values (mk-sal-inputdecl nil vardecl) db))
		((sal:output-decl? vdecl)
		 (values (mk-sal-outputdecl nil vardecl) db))
		(t (values (mk-sal-globaldecl nil vardecl) db)))))))

(defmethod abstract-on-pols* ((idecl invar-decl) db)
  (values nil db))

(defmethod abstract-on-pols* ((defn def-decl) db)
  (declare (special *amap* *in-amap*))
  (let ((amap (append *amap* *in-amap*)))
    (multiple-value-bind (adefn-list db1)
	(abstract-on-pols-defn-list (definitions defn) db amap nil)
      (values (apply #'mk-sal-defdecl nil adefn-list) db1))))

(defun abstract-on-pols-defn-list (def-list db amap answer)
  (abstract-on-pols-defn-list* def-list db amap answer))

(defun abstract-on-pols-defn-list* (def-list db amap answer)
  (declare (special *symtab* *dlevel*))
  (if (null def-list) (values answer db)
      (let ((lhs (lhs (car def-list)))
	    (rhs (rhs-definition (car def-list))))
	 (cond ((sal:tc-eq lhs 'sal::BOOLEAN)
		(multiple-value-bind (afml1 db1) 
		    (hsal-abs-fmla:sal-abstract-formula rhs db *symtab* amap)
		  (abstract-on-pols-defn-list* (cdr def-list) db1 amap
			(cons (mk-sal-simpledefinition nil lhs (afmla2sal afml1)) answer))))
		(t (print-debug 10 t "Unable to handle DEFINITIONS to nonbooleans~%") (break))))))

(defmethod abstract-on-pols* ((idecl initfor-decl) db)
  (declare (special *symtab* *amap* *in-amap*))
  (let* ((fml0 (expression idecl)))
  	 ; (fml1 (cdr (assoc fml0 *symtab* :test #'eq)))
    (assert (or (not idecl) fml0))
    (multiple-value-bind (afml1 db1) 
	(hsal-abs-fmla:sal-abstract-formula fml0 db *symtab* (append *amap* *in-amap*))
      (values (apply #'mk-sal-initdecl nil (andOfEqns2defnList* afml1 nil))
	      db1))))

;; symtab does not change hereafter. 
(defmethod abstract-on-pols* ((idecl init-decl) db)
  (let ((idecls (sal:definitions-or-commands idecl)))
    (multiple-value-bind (aidecls db1)
		(if (every #'sal:simple-definition? idecls)
		    (abstract-initialization idecls db)
		    (abstract-trans idecls db))
      (if (guarded-command? (car idecls))
    	  (values (mk-sal-initdecl nil 
		    (apply #'mk-sal-somecommands nil aidecls)) db1)
    	  (values (apply #'mk-sal-initdecl nil aidecls) db1)))))

(defmethod abstract-on-pols* ((tdecl trans-decl) db)
  (declare (special *dlevel*))
  (print-debug 5 t "Abstracting TRANSITION DECL.~%")
  (multiple-value-bind (atdecls db1) 
		       (abstract-trans (definitions-or-commands tdecl) db)
    (print-debug 5 t "Abstracting TRANSITION DECL Done.~%")
    (if (guarded-command? (car atdecls))
    	(values (mk-sal-transdecl nil (apply #'mk-sal-somecommands nil atdecls)) db1)
    	(values (apply #'mk-sal-transdecl nil atdecls) db1))))

;; (defun set-union (a b &optional (res nil) &key (test #'eq)))
(defun set-union (a b &optional (res nil) &key (test
      #'(lambda(x y) (or (eq x y) (eq (prep:polyrepConstMultiple? x y) 1)))))
  (cond ((and (null a) (null b))
	 res)
	((null a)
	 (set-union b nil res :test test))
	(t
	 (if (member (car a) res :test test)
	     (set-union (cdr a) b res :test test)
	     (set-union (cdr a) b (cons (car a) res) :test test)))))

(defun get-abstract-var-decls (amap in-map inout-map)
  (let ((sigtype (mk-sal-typename nil 'SIGN)))
    (nconc
      (loop for i in amap if (not (member i inout-map)) collect
	     ;;(mk-sal-localdecl nil (mk-sal-vardecl nil (car i) sigtype))
	     (mk-sal-globaldecl nil (mk-sal-vardecl nil (car i) sigtype)))
      (loop for i in in-map collect
	     (mk-sal-inputdecl nil (mk-sal-vardecl nil (car i) sigtype)))
      (loop for i in amap if (member i inout-map) collect
	     (mk-sal-outputdecl nil (mk-sal-vardecl nil (car i) sigtype))))))

(defun get-abstraction-mapping (pols fn oldmap chk? &optional (amap nil))
  (if (null pols) amap
      (let* ((p0 (funcall fn pols))		;; fn = caddar
	     (p1 (chaining-dp::normalizePoly p0))	;; nnormalize->
	     (test #'(lambda(x y)(or (eq x y) 
				     (eq (prep:polyrepConstMultiple? x y) 1))))
	     (old (rassoc p1 oldmap :test test))
	     (done (if (and (null old) chk?) (rassoc p1 amap :test test) old))
	     (name (if (null done) (mk-sal-nameexpr nil (intern (gensym)))))
	     (amap1 (if done (cons done amap) (acons name p0 amap))))
	;; NOTE: need p0 above; s.t. "eq" preserved between saturated pols
	(get-abstraction-mapping (cdr pols) fn oldmap chk? amap1))))
;; ============================================================================

;; ============================================================================
;; GENERIC: abstract-trans: obj * database-of-old-results --> 
;;	abstract-obj*new-database.
;; ============================================================================
(defun abstract-trans (decls db)
  (declare (special *amap* *namap* *in-amap* *feasL* *pols* *rest* *symtab* *assgns* 
		    *INV-call* *E0* *R0* *S0* *current-sal-module*))
  (cond ((sal.hsal:piecewise-continuous? decls) 	;; CHECK BELOW *in-amap*
	 (sal.hsal.pl:abstract-piecewise-system *current-sal-module* decls *amap* *namap* *in-amap* *feasL*
		*pols* *rest* *assgns* *INV-call* *symtab* db *E0* *R0* *S0*))
	((sal.hsal:hybrid-automaton? decls)
	 (abstract-hybrid-trans decls db))
	(t ;(format t "Unknown Hybrid Systems: Assuming DTS~%")
	 (abstract-hybrid-trans decls db))))

(defun abstract-hybrid-trans (decls db &optional (res nil))
  (if (null decls) (values (nreverse res) db)
      (multiple-value-bind (adecl1 db1) (abstract-hybrid-tran* (car decls) db)
	(abstract-hybrid-trans (cdr decls) db1 (cons adecl1 res)))))

(defmethod abstract-hybrid-tran* ((tran guarded-command) db)
  (declare (special *dlevel*))
  (print-debug 8 t "Abstracting the transition ~A.~%" tran)
  (if (sal.hsal:continuous-tran? tran)
      (abstract-cont-tran1 tran db) 	;; continuous transition
      (abstract-dist-tran1 tran db))) 	;; discrete transition

(defmethod abstract-hybrid-tran* ((sd simple-definition) db)
  (multiple-value-bind (dass cass) (partition-assgns (list sd))
    (declare (ignore cass))
    (if dass (values (car dass) db)
	(sal-error sd "PANIC: Can only handle guarded commands right now~%"))))

(defmethod abstract-hybrid-tran* ((other t) db)
  (sal-error other "PANIC: Can only handle guarded commands right now~%")
  (values other db))

(defun abstract-initialization (decls db)
  (declare (special *amap* *dlevel* *E0* *R0* *S0*))
  (multiple-value-bind (dass cass) (partition-assgns decls)
      (let* ((var2defn (mapcar #'(lambda(x) ;; nil shd be *this-state*!!
				   (let* ((e0 (rhs-definition x))
	 				  (e3 (pvs2prep:pvs2polyrep e0)))
				     (cons (lhs x) e3))) cass))
	     (adecls (loop for (av . qx) in *amap* collect
			(mk-sal-simpledefinition nil av (mk-sal-nameexpr nil
			  (let ((qfx (prep:applySubstitution qx var2defn)))
			    (infer-sign qfx *E0* *R0* *S0*)))))))
	(values (nconc adecls dass) db))))
				
;; ============================================================================

;; ============================================================================
;; Abstract continuous transition "tran" using "db".
;; return a list of sal-simpledefinitions
;; ============================================================================
(defun abstract-cont-tran1 (tran db)
  (declare (special *pols* *amap*))
  (assert *pols*)
  (multiple-value-bind (agua1 db1) (abstract-on-pols* (guard tran) db)
    (let* ((agua2 (sal.hsal:mk-sal-prime agua1 (mapcar #'car *amap*)))
	   ;; 11.09.10: abstract-cont-tran1: (guard AND guard') to ensure next-state in state-inv
	   (assgn (get-qualitative-next-assgn *pols* tran ))
           (insal (apply #'mk-sal-assignments nil assgn)))
      (values (mk-sal-guardedcommand nil agua2 insal) db1))))

(defun get-qualitative-next-assgn (pollist tran &optional (res '()))
  (declare (special *pols* *rest* *amap* *namap* *feasL* *assgns* *symtab* *E0* *R0* *S0* *current-sal-module*))
  (if (null pollist) 
      res
      (let* ((p1 (caddar pollist))
	     (p1dot (poltran2pdot p1 tran *pols*))
	     (p2dot (poltran2pdot p1 tran *rest*))
	     (a1 (car (rassoc p1 *amap*))))
	(cond ((and (null p1dot) (null p2dot)) 
	       (let* ((flow (sal.hsal:get-flow *current-sal-module* 
				(assignments tran)))
		      (p1dot (prep:derivativeNewPol p1 flow))
		      (Ac123 (hsal-abs-real:sal-abstract-expr-real p1dot tran *symtab* 
				*amap* *namap* *feasL* *E0* *R0* *S0*)))
      		 (setf (nth (if (car Ac123) 4 5) *assgns*) t)
	         (get-qualitative-next-assgn (cdr pollist) tran 
			(cons (sal.hsal:mk-sal-quant-assgn5 a1 Ac123) res))))
	      ((and (null p1dot) (numberp (car p2dot)))
	       (setf (nth 0 *assgns*) t)
	       (get-qualitative-next-assgn (cdr pollist) tran 
			(cons (sal.hsal:mk-sal-quant-assgn2 a1 (car p2dot)) res)))
	      (t
	       (let* ((pdot (if p1dot p1dot p2dot))
		      (a1dot (car (rassoc (car pdot) *amap*)))
		      (sign  (if p1dot t (eq (cadr p2dot) 1))))
  		 (setf (nth (if sign 0 1) *assgns*) t)
		 (get-qualitative-next-assgn (cdr pollist) tran 
			(cons (sal.hsal:mk-sal-quant-assgn3 a1 a1dot sign) res))))))))

(defun poltran2pdot (pol tran ptpdotlist &optional (fn #'eq))
  (loop for i in ptpdotlist thereis 
	(when (and (apply fn (list pol (car i))) 
		   (apply fn (list tran (cadr i)))) (cddr i))))

(defmethod abstract-on-pols* ((guard guard) db)
  (declare (special *symtab* *amap* *in-amap* *INV-call*))
  (let* ((expr (expression guard)))
	 ; (gua2 (cdr (assoc expr *symtab* :test #'eq)))
    (multiple-value-bind (agua1 db1) 
	(hsal-abs-fmla:sal-abstract-formula expr db *symtab* (append *amap* *in-amap*))
      (values (mk-sal-guard nil 
		(if *INV-call* (sal.hsal:mk-sal-bool (afmla2sal agua1) *INV-call* sal:'AND)
		    (afmla2sal agua1))) db1))))
;; ============================================================================

;; ============================================================================
;; Abstract discrete transitions, possibly with resets!
;; ============================================================================
(defun abstract-dist-tran1 (tran db)
  (multiple-value-bind (aguard db1) (abstract-on-pols* (guard tran) db)
  (multiple-value-bind (dass cass) (partition-assgns (assignments tran))
  (multiple-value-bind (aass2 db2) (abstract-cont-assigns cass db1 tran)
    (values (mk-sal-guardedcommand nil aguard
		(apply #'mk-sal-assignments nil (nconc dass aass2))) db2)))))

;; Algo: get-changed-variables, get-changed-polynomials, 
;; for each p in changed-polys, abstract-expression p(f(x)) under context "c", 
;; to get conditions (c1,c2,c3), and return: IF c1 THEN pos ELSIF c2 THEN 
;; zero ELSIF c3 THEN neg ELSE {pos,neg,zero}
(defun abstract-cont-assigns (decls db tran)
  (declare (special *amap*))
  (if (null decls) (values nil db)
      (let* ((var2defn (mapcar #'(lambda(x) ;; nil shd be *this-state*!!
				   (let* ((e0 (rhs-definition x))
	 				  (e3 (pvs2prep:pvs2polyrep e0)))
				     (cons (name (lhs x)) e3))) decls))
	     (changedVars (mapcar #'car var2defn))
	     (changedPols (loop for i in *amap* 
			   if (prep:contains? (cdr i) changedVars) collect i)))
	(abstract-cont-assignments* var2defn changedPols db tran))))

;; loop over changedPols to create the new assignments
(defun abstract-cont-assignments* (var2defn changedPols db tran &optional (ans '()))
  (declare (special *assgns* *symtab* *amap* *namap* *feasL* *E0* *R0* *S0*))
  (if (null changedPols) (values ans db)
      (let* ((vp (car changedPols))
	     (av (car vp))	;; abstract variable
	     (qx (cdr vp))	;; concrete polynomial in newrep
	     (qfx (prep:applySubstitution qx var2defn))
	     (Ac123 (hsal-abs-real:sal-abstract-expr-real qfx tran *symtab* *amap* 
			*namap* *feasL* *E0* *R0* *S0*))
	     (newdefn (sal.hsal:mk-sal-quant-assgn4 Ac123 av)))
    	(setf (nth (if (car Ac123) 2 3) *assgns*) t)
	(abstract-cont-assignments* var2defn (cdr changedPols) db tran (cons newdefn ans)))))

(defun partition-assgns (alist &optional (dass nil) (cass nil))
  (if (null alist) (values dass cass)
      (if (sal:tc-eq (sal:lhs (car alist)) 'sal::REAL)
	  (partition-assgns (cdr alist) dass (cons (car alist) cass))
	  (partition-assgns (cdr alist) (cons (car alist) dass) cass))))

;; ============================================================================
;; Assume cmodule is recursive sync-composition of basemodules.
;; ============================================================================
(defmethod get-all-components* ((cmod synchronous-composition))
  (nconc (get-all-components* (module1 cmod))
	 (get-all-components* (module2 cmod))))

(defmethod get-all-components* ((cmod asynchronous-composition))
  (nconc (get-all-components* (module1 cmod))
	 (get-all-components* (module2 cmod))))

(defmethod get-all-components* ((mod-inst module-instance))
  (get-all-components* (module (mod-decl (mod-name mod-inst)))))

(defmethod get-all-components* ((basemodule base-module))
  (list basemodule))

(defmethod get-all-components* ((module t))
  (sal-error module "Error: Cannot mix || operator with rest~%"))
;; ============================================================================

;; ============================================================================
;; Return Input,Local,Output var ids for this basemodule
;; ============================================================================
(defun get-ILOG-list (module)
  (let* ((idecls (sal:input-decls module))
	 (odecls (sal:output-decls module))
	 (ldecls (sal:local-decls module))
	 (gdecls (sal:global-decls module))
	 (iRiB (partition-on-types idecls))
	 (ideclsR (car iRiB))
	 (ideclsB (cdr iRiB))
	 (lRlB (partition-on-types ldecls))
	 (ldeclsR (car lRlB))
	 (ldeclsB (cdr lRlB))
	 (oRoB (partition-on-types odecls))
	 (odeclsR (car oRoB))
	 (odeclsB (cdr oRoB))
	 (gRgB (partition-on-types gdecls))
	 (gdeclsR (car gRgB))
	 (gdeclsB (cdr gRgB)))
    (list ideclsB ldeclsB odeclsB gdeclsB ideclsR ldeclsR odeclsR gdeclsR)))
  
(defun partition-on-types (vardecls)
  (let ((vdeclsR (loop for i in vardecls 
			if (sal:tc-eq i 'sal::REAL) collect (id i)))
  	(vdeclsB (loop for i in vardecls 
			if (not (sal:tc-eq i 'sal::REAL)) collect (id i))))
    (cons vdeclsR vdeclsB)))

(defun get-all-variables (ilog-list)
  (let* ((lvars (loop for i in ilog-list append (nth 5 i)));; local real vars
	 (ovars (loop for i in ilog-list append (nth 6 i)))
	 (all1  (append lvars ovars))
	 (gvars (loop for i in ilog-list append (nth 7 i)))
	 (ivars (loop for i in ilog-list append (nth 4 i)))
	 (all2 (append-if-new all1 gvars))
	 (all3 (append-if-new all2 ivars)))
    (loop for i in all3 if (not (sal.hsal:dummy-variable? (sal:id i))) collect i)))

(defun append-if-new (oldlist newlist)
  (if (null newlist) oldlist
      (if (member (id (car newlist)) oldlist :key #'id) 
	  (append-if-new oldlist (cdr newlist))
	  (append-if-new (cons (car newlist) oldlist) (cdr newlist)))))
;; ============================================================================

;; ============================================================================
;; get all polynomials in guards etc containing the input variables
;; ============================================================================
(defun get-seed-pols (module ilog db seed)
  (declare (special *dlevel*))
  (let* ((decls (declarations module))
	 ;(invariant (find-if #'invar-decl? decls)) 
	 (initfmla (find-if #'initfor-decl? decls))
	 (init (find-if #'init-decl? decls))
	 (def-decl (find-if #'def-decl? decls))
	 (initpols (if init (loop for i in (definitions-or-commands init)
			if (guarded-command? i) collect (expression (guard i)))))
	 (tdecl (find-if #'trans-decl? decls))
	 (cmds (if tdecl (definitions-or-commands tdecl)))
	 (tranpols (cond ((and cmds (sal.hsal:hybrid-automaton? cmds))
			  (sal.hsal:get-guards cmds))
			 ((and cmds (sal.hsal:piecewise-continuous? cmds))
			  (sal.hsal:get-rhs-exprs cmds))
			 ((every #'sal:guarded-command? cmds)
			  (sal.hsal:get-guards cmds))
			 ((every #'sal:simple-definition? cmds)		;; Environment (DTS).
			  nil)
			 (t (print-debug 10 t "Unknown Hybrid System: Check.. ~%") (break))))
	 (def-pols (if def-decl (sal.hsal:get-rhs-exprs (definitions def-decl))))
	 (stack0 (nconc initpols tranpols def-pols))
	 (stack (if initfmla (cons (expression initfmla) stack0) stack0))
	 (allpols0 (hsal-abs-db:getRAFsFromDB (nconc stack seed) db))
	 (allpols1 (delete-if #'prep:polyrepConstant? allpols0))
	 (inputvars (nth 4 ilog))
	 ;(allpols0 (loop for i in db if (RAF-p (cdr i)) collect 
			;(RAF-pol (cdr i))))
	 (allpols2 (list2set allpols1))
	 (inpols (getExactlyRelevantPols1 allpols2 inputvars))
	 (restpols (set-difference allpols2 inpols)))
    ;(break)
    (cons inpols restpols)))
  
(defun getExactlyRelevantPols1 (plist vlist &key (key #'(lambda(x) x)))
  (remove-if-not #'(lambda (x) (relevant? x vlist)) plist :key key))

(defun relevant? (pol vars)
  (subsetp (prep:allVarsIn pol) vars :test #'prep:var-equal?))

(defun getWeaklyRelevantPols1 (plist vlist &key (key #'(lambda(x) x)))
  (remove-if-not #'(lambda (x) (weaklyRelevant? x vlist)) plist :key key))

(defun weaklyRelevant? (pol vars)
  (and (not (relevant? pol vars))	;; not strongly relevant, but relevant!
       (intersection (prep:allVarsIn pol) vars :test #'prep:var-equal?)))

(defun list2set (in-list &optional (out-set nil) &key (test #'eq))
  (if (null in-list) out-set 
      (list2set (cdr in-list) 
  		(if (member (car in-list) out-set :test test) 
		    out-set (cons (car in-list) out-set)))))
;; ============================================================================

;; ============================================================================
;; mk-hybrid-mk-sal-misc.lisp
;; ============================================================================
(defun andOfEqns2defnList* (afmla sgn)
  (declare (special *dlevel*))
  (cond ((DNF-p afmla)
  	 (let ((products (DNF-products afmla)))
    	   (if  (or (null products) (cdr products))
  		(sal-error t "Can not represent the abstract simple definition~%")
		(loop for i in (car products) nconc (andOfEqns2defnList* i sgn)))))
	((sal::infix-application? afmla)
  	 (let* ((lhs (sal::args1 afmla))
	 	(rhs (sal::args2 afmla)))
    	   (cond ((eq (id (sal:operator afmla)) sal:'AND)
	   	  (nconc (andOfEqns2defnList* lhs sgn) 
		  	 (andOfEqns2defnList* rhs sgn)))
          	 ((eq (id (sal:operator afmla)) '=)
	   	  (list (if sgn 
	      		(mk-sal-simpledefinition nil (mk-sal-nextoperator nil lhs) rhs)
	      		(mk-sal-simpledefinition nil lhs rhs))))
          	 ((eq (id (sal:operator afmla)) '/=)
	   	  (list (mk-sal-simpledefinition nil 
			  (if sgn (mk-sal-nextoperator nil lhs) lhs)
			  (mk-sal-rhsselection nil 
	                    (apply #'mk-sal-setlistexpression nil (get-others rhs))))))
		 ((eq (id (sal:operator afmla)) sal:'OR)
		  (print-debug 10 t "WARNING: Initial conditions are a SUBSET of actuals.~%")
	   	  (andOfEqns2defnList* lhs sgn))
		  	 ;; (andOfEqns2defnList* rhs sgn)
		 (t
           	  (sal-error t "Can't represent the abstract simple definition~%")))))
	((sal::name-expr? afmla)
	 (if (eq (sal::id afmla) 'sal:FALSE)
	     (print-debug 10 t "WARNING: FALSE assignment? CHECK Initialization in abstraction~%"))
	 nil)	;; TRUE or FALSE.
	(t
         (sal-error t "Can't represent the abstract simple definition~%"))))

(defun afmla2sal (afmla)
  (cond ((DNF-p afmla)
	 (let* ((dnf (DNF-products afmla))
		(dnf1 (loop for i in dnf collect
			(loop for j in i collect (afmla2sal j))))
		(sal-ands (loop for i in dnf1 collect
				(sal.hsal:mk-syntactic-bool* i sal:'AND))))
	   (sal.hsal:mk-syntactic-bool* sal-ands 'OR)))
	((expression? afmla)
	 afmla)
	(t (sal-error t "Unreachable code:~A~%" afmla))))

(defun get-others (rhs) 
  "rhs is a sal nameexpr: either pos, neg, or zero"
  (assert (sal::name-expr? rhs))
  (cond ((eq rhs sal.hsal:sal-pos) 
	 (list sal.hsal:sal-neg sal.hsal:sal-zero))
	((eq rhs sal.hsal:sal-neg) 
	 (list sal.hsal:sal-pos sal.hsal:sal-zero))
	((eq rhs sal.hsal:sal-zero) 
	 (list sal.hsal:sal-pos sal.hsal:sal-neg))
	(t (sal-error t "Unreachable code reached:~A~%" (sal::id rhs)) (break))))
;; ============================================================================

;; ============================================================================
;; Random Collection of missing functions.
;; ============================================================================
;; get all parameters (ids) in this module
(defun get-parameters (module)
  (declare (special sal::*context*))
  (let* ((mid (id (mod-name (modref module))))
	 (this-module? #'(lambda(x) (and (module-declaration? x) (eq (id x) mid))))
	 (module-decl (find-if this-module? (context-body sal::*context*))))
    (loop for y in (parameters module-decl) collect (id y))))
;; ============================================================================

;; ============================================================================
;; Dealing with the invariant formula.
;; Input: Module whose invariant set is to be saturated.
;; Output: (E R) where (E R) is a saturation of (E0, R0)
;; ============================================================================
(defun saturate-invariant-set (module symtab lorder lparams)
  (declare (special *dlevel*))
  ;; (format t "Saturating the invariant set, creating *E0*, *R0*, *S0*...~%")
  (prep:set-variables lorder)
  (prep:set-parameters lparams)
  (let* ((decls (declarations module))
	 (invar (find-if #'invar-decl? decls))
	 (expr1 (if invar (expression invar) nil)))
    (if (and invar expr1)
	(let* ((a1234 (hsal-abs-real:pvsfml2polyrep expr1 symtab)) 
	       (eqns (car a1234))
	       (rules (cadr a1234))
	       (greatereqs (caddr a1234))
	       (others (cadddr a1234)))
	  (if others
	      (sal-error expr1 "Can't handle complex invariant fmlas~%"))
	  (multiple-value-bind (st E1 R1 S1)
	  	(chaining-dp:saturate eqns rules greatereqs)
            (print-debug 5 t "Invariant Saturated.~%E=~A~%R=~A~%S=~A~%"
		(chaining-dp::dpPrint E1) (chaining-dp::dpPrint R1) (chaining-dp::dpPrint S1))
	    ;; (break)
	    (if (null st) ;; (eq st chaining-dp:'inconsistent)
		(progn (print-debug 10 t "Invariant Formula is inconsistent!~%") 
		       (print-debug 10 t "Witness: ~A~%" E1) (break)))
	    (loop for i in E1 do (setf (chaining-dp::POL-wit i) nil) (setf (chaining-dp::POL-op i) '=))
	    (loop for i in R1 do (setf (chaining-dp::POL-wit i) nil) (setf (chaining-dp::POL-op i) '>))
	    (loop for i in S1 do (setf (chaining-dp::POL-wit i) nil) (setf (chaining-dp::POL-op i) '>=))
	    (list E1 R1 S1)))
	(list nil nil nil))))
;; ============================================================================

;; ============================================================================
;; Return the property from the declarations inside the given context
;; ============================================================================
(defun get-property (c property)
  (if (null property) (return-from get-property 'nil))
  (if (eq property sal::'all) (return-from get-property
      (remove-if-not #'assertion-declaration? (context-body c))))
  (let* ((ctxt-body (context-body c))
	 (testfn #'(lambda (x y) (and (assertion-declaration? y)
				      (eq (id y) x))))
         (ans (find property ctxt-body :test testfn)))
    (if ans (list ans))))

(defun polynomials-in-property (property)
  (loop for i in property nconc (sal:pvsexpr2RAFstrings (assertion (assertion i)))))
;; ============================================================================

;; ============================================================================
;; Abstract the given property w.r.t the given polynomials.
;; Copied from sal-hybrid-new-new.lisp.abstract-on-pols.
;; Algo: Go down, print SAL, when RAF, get absvar from amap.
;; ============================================================================
(defun abstract-property-on-pols (prop amap order params)
  (loop for i in prop collect (abstract-property-on-pols* i amap order params)))

(defun abstract-property-on-pols* (prop amap order params)
  (let* ((propId (id prop))			;; symbol
	 (asForm (assertion-form prop))		;; symbol
	 (modmodels (assertion prop))
	 (modId (id (mod-name (module modmodels))))
	 (fmla  (assertion modmodels))
	 (afmla (abstract-property-on-pols1 fmla amap order params))
         (aafmla (mk-sal-modulemodels nil 
			(mk-sal-moduleinstance nil 
			  (mk-sal-modulename nil modId)
			  (mk-sal-moduleactuals nil))
			afmla)))
    (mk-sal-assertiondeclaration nil propId
			(mk-sal-assertionform nil asForm)
			aafmla)))

(defun abstract-property-on-pols1 (fmla amap order params)
  (prep:set-variables order)
  (prep:set-parameters params)
  (let* ((*amap* amap))
    (declare (special *amap*))
    (abstract-property-on-pols1S2B* fmla)))

(defmethod abstract-property-on-pols1S2B* ((fmla sal::application))
  (let ((ope (sal:operator fmla))
	(arg (sal:argument fmla)))
    (if (or (eq (id ope) sal::'G) (eq (id ope) sal::'F))
	(mk-sal-application nil (mk-sal-nameexpr nil (id ope))
			        (abstract-property-on-pols1S2B* arg))
	(sal-error t "PANIC: Unknown LTL operator~%"))))

(defmethod abstract-property-on-pols1S2B* ((fmla sal::tuple-literal))
  (apply #'sal:mk-sal-tupleliteral nil 
    (mapcar #'abstract-property-on-pols1B* (exprs fmla))))

(defmethod abstract-property-on-pols1B* ((fmla sal::application))
  (case (id (sal:operator fmla))
	((sal:AND sal:OR sal::=> sal::<=>) 
  	 (mk-sal-application '(sal:infix? t)
		(mk-sal-nameexpr nil (id (sal:operator fmla)))
		(abstract-property-on-pols1B* (sal:argument fmla))))
	((sal::NOT sal::G sal::F)
  	 (mk-sal-application nil
		(mk-sal-nameexpr nil (id (sal:operator fmla)))
		(abstract-property-on-pols1B* (sal:argument fmla))))
	((< <= >= > =)
	 (if (sal:tc-eq (sal:args1 fmla) 'sal::BOOLEAN)
	     (mk-sal-application '(sal:infix? t)
		(mk-sal-nameexpr nil (id (sal:operator fmla)))
		(abstract-property-on-pols1B* (sal:argument fmla)))
	     (abstract-property-on-pols1AF fmla)))
	(t (sal-error t "PANIC: What the hell is this expr ~A~%" fmla))))

(defmethod abstract-property-on-pols1B* ((fmla sal::tuple-expr))
  (apply #'mk-sal-tupleliteral nil
    (loop for i in (sal:exprs fmla) collect (abstract-property-on-pols1B* i))))

(defmethod abstract-property-on-pols1B* ((fmla t))	;; TRUE/FALSE
  fmla)

(defun abstract-property-on-pols1AF (fmla)
  (declare (special *amap*))
  (let* ((pol1 (pvs2prep:pvs2polyrep (sal:args1 fmla)))
	 (pol2 (pvs2prep:pvs2polyrep (sal:args2 fmla)))
	 (op (id (sal:operator fmla)))
	 (pol3 (prep:polyrepAddPoly pol1 (prep:polyrepNegativePoly pol2)))
	 (ans (loop for i in *amap*
		    for a = (prep:polyrepConstMultiple? pol3 (cdr i))
		thereis (when a (cons (car i) a))))
	 (absvar (if (null ans) (break) (car ans)))
	 (swap (eq (cdr ans) -1)))
    (cond ((and (null pol3) (eq op '=))
	   (sal:mk-sal-nameexpr nil 'TRUE))
	  ((and (null pol3))
	   (sal:mk-sal-nameexpr nil 'FALSE))
	  ((or (and (eq op '>) (not swap)) (and (eq op '<) swap))
	   (sal.hsal:mk-sal-equation absvar sal.hsal:sal-pos))
	  ((or (and (eq op '<) (not swap)) (and (eq op '>) swap))
	   (sal.hsal:mk-sal-equation absvar sal.hsal:sal-neg))
	  ((eq op '=)
	   (sal.hsal:mk-sal-equation absvar sal.hsal:sal-zero))
	  ((or (and (eq op '<=) (not swap)) (and (eq op '>=) swap))
	   (sal.hsal:mk-sal-bool (sal.hsal:mk-sal-equation absvar sal.hsal:sal-neg) 
			(sal.hsal:mk-sal-equation absvar sal.hsal:sal-zero) 'OR))
	  (t
	   (sal.hsal:mk-sal-bool (sal.hsal:mk-sal-equation absvar sal.hsal:sal-pos) 
			(sal.hsal:mk-sal-equation absvar sal.hsal:sal-zero) 'OR)))))
;; ============================================================================

;; ============================================================================
(defun get-feasible-states (pols namap E0 R0 S0 amap)
  (declare (special *dlevel*))
  (if (null pols) (return-from get-feasible-states nil))
  (dp-feas:set-feas-debug-level *dlevel*)
  (let ((feasL (dp-feas:get-feasible-states pols namap E0 R0 S0)))
      ;; (format t "FEASL = ~A~%" feasL)
      ;; (break)
      ;; (if (and (every #'null ninfeas) (every #'null feas))
	    ;; (return-from get-feasible-states nil))
      (let ((ans (mk-sal-feas1 feasL amap namap)))
	(multiple-value-bind (INV-call INV-decl) (get-INV-declaration amap ans)
	  (values INV-call INV-decl feasL)))))

(defun mk-sal-feas1 (feasL amap namap)
  (let ((inv-parts (loop for f in feasL collect 
			 (mk-sal-feas2 (dp-feas:FEAS-ff f) (dp-feas:FEAS-tt f) 
				       (dp-feas:FEAS-pols f) amap namap))))
    (sal.hsal:mk-syntactic-bool inv-parts sal:'AND)))

(defun mk-sal-feas2 (infeas feas pols amap namap)
  (multiple-value-bind (op parts)
  	(if (or (eq feas -1) (and (not (eq infeas -1)) (< (length infeas) (length feas))))
	    (values sal:'AND (loop for i in infeas collect (mk-sal-infeas3 i amap namap)))
	    (values sal:'OR (loop for i in feas collect (mk-sal-feas3 i pols amap namap))))
    (sal.hsal:mk-syntactic-bool parts op)))

(defun mk-sal-infeas3 (infeas amap namap)
  (let ((parts (loop for i in infeas collect 
		(mk-sal-atomic-fmla  (chaining-dp::POL-pol i) (chaining-dp::POL-op i) amap namap))))
    (mk-sal-application nil (mk-sal-nameexpr nil sal:'NOT) (mk-sal-tupleliteral nil (sal.hsal:mk-syntactic-bool parts sal:'AND)))))

(defun mk-sal-feas3 (feas pols amap namap)
  (let ((parts (loop for i in feas as j in pols collect (mk-sal-atomic-fmla  j i amap namap))))
    (sal.hsal:mk-syntactic-bool parts sal:'AND)))

(defun mk-sal-atomic-fmla (pol op amap namap)
  (let ((avar (car (rassoc pol amap)))
	(aval (case op  (> sal.hsal:sal-pos) 
			(< sal.hsal:sal-neg) 
			(= sal.hsal:sal-zero) 
			(t (format t "ERROR!!!~%")(break)))))
    (if avar (sal.hsal:mk-sal-equation avar aval)
  	(let* ((npol (car (rassoc pol namap)))
	       (avar (car (rassoc npol amap)))
	       (bval (case op 	(> sal.hsal:sal-neg) 
				(< sal.hsal:sal-pos) 
				(= sal.hsal:sal-zero) 
				(t (format t "ERROR!!!~%")(break)))))
    	  (if avar (sal.hsal:mk-sal-equation avar bval)
  	      (multiple-value-bind (avar sgn) 
		(loop for i in amap for j = (prep:polyrepConstMultiple? pol (cdr i)) if j return (values (car i) j))
    	  	(if avar (sal.hsal:mk-sal-equation avar (if (eq sgn 1) aval bval))
	      	    (progn 
		     (format t "******THIS SHOULD BE UNREACHABLE CODE******.~%")
		     (break)))))))))

(defun get-INV-declaration (amap value)
  (multiple-value-bind (formal-vars real-vars)
	(mk-sal-dummy-vardecls amap)
    (let ((newINV (intern (gensym "INV"))))
      (values (mk-sal-application nil (mk-sal-nameexpr nil newINV)
		(apply #'mk-sal-tupleliteral nil (mapcar #'(lambda(x)(mk-sal-nextoperator nil x)) real-vars)))
    	      (mk-sal-constantdeclaration nil newINV
		(apply #'mk-sal-vardecls nil formal-vars)
		(mk-sal-typename nil 'BOOLEAN) value)))))

(defun mk-sal-dummy-vardecls (amap)
  (let ((typename (mk-sal-typename nil 'SIGN)))
    (values (mapcar #'(lambda(x) (mk-sal-vardecl nil (id (car x)) typename)) amap)
	    (mapcar #'car amap))))
;; ============================================================================

;; ============================================================================
;; Test for inconsistency.
;; ============================================================================
(defun twoModes2OneMode (symtab)
  #'(lambda (tt1 ff1 tt2 ff2) (twoModes2OneMode* tt1 ff1 tt2 ff2 symtab)))

(defun twoModes2OneMode* (tt1 ff1 tt2 ff2 symtab)
  (declare (special *dlevel*))
  (if (or (and (null tt1) (null ff1)) (and (null tt2) (null ff2))) (return-from twoModes2OneMode* (values (nconc tt1 tt2) (nconc ff1 ff2))))
  (print-debug 3 t "twoMode2OneMode: called with~% tt1=~A~% tt2=~A~% ff1=~A~% ff2=~A~%" tt1 tt2 ff1 ff2)
  (multiple-value-bind (tt ff) 	;; check consistency of the boolean part
	(consistent-mode tt1 tt2 (append ff1 ff2) symtab)
      (print-debug 3 t "Modes~% tt1=~A~% tt2=~A~% ff1=~A~% ff2=~A~% was found~% tt=~A~% ff=~A~%" 
	tt1 tt2 ff1 ff2 tt ff)
      (if (eq tt -1) -1 (values tt ff))))

;; NEW NEW
(defun consistent-mode (tt1 tt2 ffs db)
  (let ((local-db1 (loop for i in tt1 collect (assoc i db)))
	(local-db2 (loop for i in tt2 collect (assoc i db)))
	(local-db3 (loop for i in ffs collect (assoc i db))))
    (multiple-value-bind (ntt nff) 
	(consistent-mode* (mapcar #'cdr local-db1) (mapcar #'cdr local-db2) (mapcar #'cdr local-db3) db)
      (let ((local-db4 (nconc local-db1 local-db2)))
        (if (eq ntt -1) -1
	    (values (loop for i in ntt collect (car (rassoc i local-db4)))
		    (loop for i in nff collect (car (rassoc i local-db3)))))))))

(defun consistent-mode* (tt1 tt2 ffs local-db)
  (let* ((n-tt2 (loop for i in tt2 collect (case (inconsistentAF* i tt1 local-db)
						 (-1 (return-from consistent-mode* -1))
						 (0 i)
						 (1 nil)
						 (t (format t "Unknown return value.~%") (break)))))
	 (tt (append tt1 (remove-if #'null n-tt2)))
	 (n-ff (inconsistentAF-neg ffs (expr2BAFRAFlist tt local-db) nil nil local-db)))
    (if (eq n-ff -1) -1 (values tt n-ff))))

;; Test if baf is consistent with all bafs. Return -1 if inconsistent, 
;; 1 if redundant, 0 otherwise.
(defun inconsistentAF* (tt ttl local-db)
  (if (null ttl) (return-from inconsistentAF* 0))
  (if (some #'(lambda(x) (and (not (RAF-p x)) (not (BAF-p x)))) ttl)
      (return-from inconsistentAF* (inconsistentAF* tt (expr2BAFRAFlist ttl local-db) local-db)))
  (if (and (not (RAF-p tt)) (not (BAF-p tt)))
      (return-from inconsistentAF* 
        (let ((ans (loop for i in (expr2BAFRAFlist (list tt) local-db) collect (inconsistentAF* i ttl local-db))))
          (if (some #'(lambda(x) (eq x -1)) ans) -1 
	    (if (every #'(lambda(x) (eq x 1)) ans) 1 0)))))
  (let ((tt1 (car ttl)))
    (cond ((and (BAF-p tt) (BAF-p tt1))
	   (if (not (prep:var-equal? (hsal-abs:BAF-var tt) (hsal-abs:BAF-var tt1)))
	       (inconsistentAF* tt (cdr ttl) local-db)
	       (if (equal (hsal-abs:BAF-val tt) (hsal-abs:BAF-val tt1)) 1 -1)))
	  ((and (RAF-p tt) (RAF-p tt1))
	   (if (eq (RAF-pol tt) (RAF-pol tt1))
	       (if (eq (RAF-op tt) (RAF-op tt1)) 1 -1)
	       (inconsistentAF* tt (cdr ttl) local-db)))
	  (t (inconsistentAF* tt (cdr ttl) local-db)))))

(defun expr2BAFRAFlist (ttl local-db &optional (ans nil))
  (if (null ttl) ans
      (let ((tt1 (car ttl)))
	(cond ((or (BAF-p tt1) (RAF-p tt1)) 
	       (expr2BAFRAFlist (cdr ttl) local-db (cons tt1 ans)))
	      ((DNF-p tt1)
	       (check-DNF-depth tt1)
	       (expr2BAFRAFlist (append (car (DNF-products tt1)) (cdr ttl)) local-db ans))
	      (t (expr2BAFRAFlist (cons (lookup-local-db tt1 local-db) (cdr ttl)) local-db ans))))))

(defun check-DNF-depth (tt)
  (if (cdr (DNF-products tt)) (progn (format t "Can't handle ORs in conditions~%") (break)) 1))

(defun lookup-local-db (i db)
  (let ((ans (cdr (assoc i db))))
    (if (null ans) (progn (format "~A not found in database.~%" i) (break))) ans))

;; If inconsistent, return -1; if redundant, return 1; else return CONSISTENT&non-redundant-part
(defun inconsistentAF-neg (fll ttl ffll realffll local-db)
  (if (null fll) (return-from inconsistentAF-neg realffll))
  (let* ((res1 (inconsistentAF-neg* (car fll) ttl local-db)))
    (if (eq res1 -1) (return-from inconsistentAF-neg -1))
    (if (eq res1 1) (inconsistentAF-neg (cdr fll) ttl ffll realffll local-db)
	(let ((res2 (inconsistentAF-negneg* res1 ffll)))
    	  (if (eq res2 -1) (return-from inconsistentAF-neg -1))
    	  (if (eq res2 1) (inconsistentAF-neg (cdr fll) ttl ffll realffll local-db)
	      (inconsistentAF-neg (cdr fll) ttl (cons res1 ffll) (cons (car fll) realffll) local-db))))))

(defun inconsistentAF-neg* (ff ttl local-db)
  (let* ((nff (expr2BAFRAFlist (list ff) local-db))
	 (ans (loop for i in nff collect 
		    (let ((lans (inconsistentAF* i ttl local-db)))
		      (if (null lans) (return-from inconsistentAF-neg* 1)) lans))))
    (if (every #'(lambda(x)(eq x 1)) ans) (return-from inconsistentAF-neg* -1))
    nff))

(defun inconsistentAF-negneg* (ffl ffll)
  (if (null ffll) (return-from inconsistentAF-negneg* 0))
  (if (every #'(lambda(x) (eq (inconsistentAF* x ffl nil) 1)) (car ffll))
      1
      (inconsistentAF-negneg* ffl (cdr ffll))))
;; ============================================================================

;; ============================================================================
;; Print Routines
;; ============================================================================
(defun print-ilog-list (mod-list ilog-list)
  (declare (special *dlevel*))
  (loop for i in ilog-list as j in mod-list do 
	(format t "Module ~A~%" (id (mod-name (modref j))))
	(if (nth 0 i) (print-debug 5 t "  BoolInput:~A~%" (nth 0 i)))
	(if (nth 1 i) (print-debug 5 t "  BoolLocal:~A~%" (nth 1 i)))
	(if (nth 2 i) (print-debug 5 t "  BoolOutpt:~A~%" (nth 2 i)))
	(if (nth 3 i) (print-debug 5 t "  BoolGlobl:~A~%" (nth 3 i)))
	(if (nth 4 i) (print-debug 5 t "  RealInput:~A~%" (nth 4 i)))
	(if (nth 5 i) (print-debug 5 t "  RealLocal:~A~%" (nth 5 i)))
	(if (nth 6 i) (print-debug 5 t "  RealOutpt:~A~%" (nth 6 i)))
	(if (nth 7 i) (print-debug 5 t "  RealGlobl:~A~%" (nth 7 i)))))

(defun print-seed-pols (mod-list seed-pols transfers amap-ext)
  (declare (special *dlevel*))
  (loop for i in seed-pols as j in mod-list do 
	(print-debug 5 t "Module ~A Base-Pols:~%" (id (mod-name (modref j))))
	(loop for k in (cdr i) do (print-debug 5 t " ~A~%" (prep:polyrepPrint k))))
  (if transfers (print-debug 5 t "All Transfers:~%"))
  (loop for i in transfers do (print-debug 5 t " ~A~%" (prep:polyrepPrint i)))
  (if amap-ext (print-debug 5 t "Extension amap: ~%"))
  (loop for i in amap-ext do (print-debug 5 t " ~A: ~A~%" (car i) (prep:polyrepPrint (cdr i)))))

(defun print-all-polynomials (pols others)
  (declare (special *dlevel*))
  (print-debug 5 t "Constructing abstraction over the following polynomials...~%")
  (loop for i in pols do (print-debug 5 t " ~A~%" (prep:polyrepPrint (caddr i))))
  (if others (print-debug 5 t "The derivatives of some of these polynomials are known~%"))
  (loop for i in others do (print-debug 5 t " ~A~%" (prep:polyrepPrint (car i)))))
;; ============================================================================

;; ============================================================================
;; Infer sign of a polynomial, in the given context.
;; Eg. x = 0 IMPLIES sign of x+4 is POS.
;; Used for initialization
;; ============================================================================
(defun infer-sign (pol E0 R0 S0)
  (declare (special *dlevel*))
  (cond ((null pol) 
	 'zero)
	((and (null (cdr pol)) (null (cdar pol))) 
	 (if (> (caar pol) 0) 'pos 'neg))
	(t
  (let* ((leq0 (chaining-dp:saturate nil (list pol) nil nil E0 R0 S0))
	 (npol (prep:polyrepNegativePoly pol))
	 (geq0 (chaining-dp:saturate nil (list npol) nil nil E0 R0 S0))
	 (neq0 (chaining-dp:saturate (list pol) nil nil nil E0 R0 S0)))
    (cond ((and (null leq0) (null geq0))
	   'zero)
	  ((and (null leq0) (null neq0))
	   'neg)
	  ((and (null geq0) (null neq0))
	   'pos)
	  (t 
	   (print-debug 10 t "WARNING: Initialization incomplete~%") 
	   'pos))))))
;; ============================================================================
