;; --------------------------------------------------------------------
;; HybridSAL
;; Copyright (C) 2006, SRI International.  All Rights Reserved.
;; 
;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License
;; as published by the Free Software Foundation; either version 2
;; of the License, or (at your option) any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
;; --------------------------------------------------------------------

;;;;;;;;;;;;;;;;;;;;;;;;;;* -*- Mode: Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; vim: syntax=lisp
;; hsal-abs-sat.lisp --
;; Author          : Ashish Tiwari
;; Created On      : Fri Oct 18, 2002
;; Last Modified By: Ashish Tiwari
;; Last Modified On: Fri Oct 18, 2002
;; Update Count    : 0
;; Status          : Unknown, use with caution
;;
;; HISTORY : 
;; 10.18.02: Hybrid Sal Abstractor Seed Polynomial Saturation Routines
;;	Packaging it into a package.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; ============================================================================
;; ======== Hybrid Sal Abstractor Seed Polynomial Saturation Routines =========
;; ============================================================================
(in-package "hsal-abstractor-saturation")

;;REQUIRE: pvs2polyrep:pvs2polyrep; prep:*
(defvar *maxdegree* 2)

;; ============================================================================
;; Saturate pols: New data-struct. No pvs-exprs.
;; ============================================================================
(defun saturate-pols-new (context module seed-pols depth E0 R0 S0)
  (declare (ignore context))
  (let ((*E0* E0) (*R0* R0) (*S0* S0))
    (declare (special *E0* *R0* *S0*))
  (let* ((tdecl (find-if #'trans-decl? (declarations module)))
	 (datastruct (loop for i in seed-pols collect (list i 0 i)))
	 (transdecls (definitions-or-commands tdecl))
	 (result (saturate-new-trans-pols-d module transdecls datastruct depth
			(mergeToFirst datastruct (emptyDS)))))
    (values (car result) (cdr result)))))

(defun saturate-new-trans-pols-d (module transdecl pols d kres)
  (if (eq d 0) kres
      (let* ((npols (saturate-new-trans-pols-1 module transdecl pols kres))
	     (kres1 (join npols kres)))
	(saturate-new-trans-pols-d module transdecl (car npols) (- d 1) kres1))))

(defun saturate-new-trans-pols-1 (module transdecls pols kres &optional (res (emptyDS)))
  (if (null transdecls) res
      (let* ((kres1 (join kres res))
	     (npols (saturate-new-tran-pols-1 module (car transdecls) pols kres1)))
	(cond ((empty? npols)		;; discrete transition
	       (saturate-new-trans-pols-1 module (cdr transdecls) pols kres res))
	      (t
	       (saturate-new-trans-pols-1 module (cdr transdecls) pols kres
					(join npols res)))))))
  
(defun saturate-new-tran-pols-1 (module transdecl pols kres &optional (res (emptyDS)))
  (if (not (sal.hsal:continuous-tran? transdecl)) nil		;; discrete transition/simple-defns
      (let ((flow (sal.hsal:get-flow module (assignments transdecl))))
	(saturate-new-tran-pols-1* transdecl flow pols kres res))))

(defun saturate-new-tran-pols-1* (transdecl flow pols kres res)
  (if (null pols) res
      (let* ((pol (caddar pols))
	     (pdot (saturate-new-tran-pol-1 flow pol)))
        (if pdot (format t " Derivative of ~A is ~A.~%" (prep:polyrepPrint pol) (prep:polyrepPrint pdot)))
	(cond ((null pdot) 		;; degree > maxdegree
	       (saturate-new-tran-pols-1* transdecl flow (cdr pols) kres res))
	      ((and (null (cdr pdot)) (null (cdar pdot)))
	       (cond ((> (caar pdot) 0)
		      (saturate-new-tran-pols-1* transdecl flow (cdr pols) kres
			(addToSecond (list pol transdecl 1 1) res)))
		     ((= (caar pdot) 0)
		      (saturate-new-tran-pols-1* transdecl flow (cdr pols) kres
			(addToSecond (list pol transdecl 0 1) res)))
		     ((< (caar pdot) 0)
		      (saturate-new-tran-pols-1* transdecl flow (cdr pols) kres
			(addToSecond (list pol transdecl -1 1) res)))))
	      (t 
	       (let ((yes (const-multiple-new-new?* pdot (car kres))))
		 (cond ((null yes)
			(saturate-new-tran-pols-1* transdecl flow (cdr pols) 
			  (addToFirst (list pol transdecl pdot) kres)
			  (addToFirst (list pol transdecl pdot) res)))
		       (t
			(saturate-new-tran-pols-1* transdecl flow (cdr pols) kres
			  (addToSecond (cons pol (cons transdecl yes)) res))))))))))

(defun const-multiple-new-new?* (npol pollist)
  (declare (special *E0* *R0* *S0*))
  (if (null pollist) (return-from const-multiple-new-new?* nil))
  (let* ((npol1 (caddar pollist))
	 (soln (prep:polyrepConstMultiple? npol npol1)))	;; npol is not null
	;(format t "Trying ~A againt ~A~%" npol npol1)
	(if soln (return-from const-multiple-new-new?* (list npol1 soln)))
	(if (and (null *E0*) (null *R0*) (null *S0*))
	    (return-from const-multiple-new-new?* 
			 (const-multiple-new-new?* npol (cdr pollist))))
	(let* ((vars (prep:polyrepGetVariables npol))
	       (vars1 (prep:polyrepGetVariables npol1))
	       (vardiff (set-difference (car vars) (car vars1)
			:test #'prep:var-equal?))
	       (vardiff1 (set-difference (car vars1) (car vars)
				:test #'prep:var-equal?)))
	  (if (or vardiff vardiff1) 
	      (return-from const-multiple-new-new?* 
			   (const-multiple-new-new?* npol (cdr pollist))))
	  (if (not (chaining-dp:check-invariant-new npol 'chaining-dp::= npol1 'chaining-dp::= *E0* *R0* *S0*))
	      (return-from const-multiple-new-new?* 
			   (const-multiple-new-new?* npol (cdr pollist))))
	  (if (chaining-dp:check-invariant-new npol 'chaining-dp::> npol1 'chaining-dp::> *E0* *R0* *S0*)
	      (if (chaining-dp:check-invariant-new npol 'chaining-dp::< npol1 'chaining-dp::< *E0* *R0* *S0*)
		  (list npol1 1)
		  (const-multiple-new-new?* npol (cdr pollist)))
	      (if (chaining-dp:check-invariant-new npol 'chaining-dp::> npol1 'chaining-dp::< *E0* *R0* *S0*)
	          (if (chaining-dp:check-invariant-new npol 'chaining-dp::< npol1 'chaining-dp::> *E0* *R0* *S0*)
		      (list npol1 -1)
		      (const-multiple-new-new?* npol (cdr pollist)))
		  (const-multiple-new-new?* npol (cdr pollist)))))))
;; ============================================================================

;; ============================================================================
;; Compute derivative of ONE poly in ONE transdecl to ONE depth.
;; Returns "nil" if "tran" is a discrete transition, and the 
;; derivative of "poly" in the continuous transition "tran" otherwise.
;; Return value is poly-in-new-rep (normalized).
;; ============================================================================
(defun saturate-new-tran-pol-1 (flow pol)
  (let* ((pdot (prep:derivativeNewPol pol flow))
	 (coef (if pdot (abs (caar pdot)) 1)))
    (cond ((> (prep:polyrepTotalDegreePoly pdot) (get-max-degree))
	   (format t " Derivative of ~A exceeds maxdegree limit, Ignoring~%" (prep:polyrepPrint pol))
	   nil)				;; degree > maxdegree
	  (pdot 
	   (prep:polyrepDividePolyCst pdot coef))
	  (t (list (list 0))))))	;; derivative is zero
;; ============================================================================

;; ============================================================================
;; Pair of list data-structure
;; ============================================================================
(defun emptyDS ()
  (cons nil nil))

(defun addToFirst (e1 p1)
  (cons (cons e1 (car p1)) (cdr p1)))

(defun addToSecond (e1 p1)
  (cons (car p1) (cons e1 (cdr p1))))

(defun join (p1 p2)
  (cons (append (car p1) (car p2))
	(append (cdr p1) (cdr p2))))

(defun mergeToFirst (l1 p1)
  (cons (append l1 (car p1))
	(cdr p1)))

(defun mergeToSecond (l1 p1)
  (cons (car p1) 
	(append l1 (cdr p1))))

(defun empty? (p1)
  (if (null p1) t
      (and (eq nil (car p1)) (eq nil (cdr p1)))))

(defun firstEmpty? (p1)
  (eq nil (car p1)))
;; ============================================================================

;; ============================================================================
;; Others can set the *maxdegree* by calling this function.
;; ============================================================================
(defun set-maxdegree (n)
  (setf *maxdegree* n))

(defun get-max-degree ()
  *maxdegree*)
;; ============================================================================
