;; --------------------------------------------------------------------
;; HybridSAL
;; Copyright (C) 2006, SRI International.  All Rights Reserved.
;; 
;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License
;; as published by the Free Software Foundation; either version 2
;; of the License, or (at your option) any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
;; --------------------------------------------------------------------

;;;;;;;;;;;;;;;;;;;;;;;;;;* -*- Mode: Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; vim: syntax=lisp
;; hsal-abs-real.lisp --
;; Author          : Ashish Tiwari
;; Created On      : Fri Oct 18, 2002
;; Last Modified By: Ashish Tiwari
;; Last Modified On: Fri Oct 18, 2002
;; Update Count    : 0
;; Status          : Unknown, use with caution
;;
;; HISTORY : 
;; 10.18.02: Abstracting Real Expression.
;;  Deduce conditions c1,c2,c3 under which expr is pos, neg, zero.
;; 11.05.02: sal-abstract-expr-real p: p can be zero/constant too!
;; 11.20.02: INVARIANT information is not used here!!!!
;; 06.18.02: sal-abstract-expr-real: E R S should be produced by chainnig-dp.
;; 07.09.03: Changed sal-abstract-expr-real1 : more efficient, 3-calls to 1-call.
;; 07.22.03: Line131: eq replaced by prep:polyrepEqual?. ***undone***FUTURE***
;; 08.25.03: chaining-dp:reset-wit added.
;; 09.02.03: allVarsPars introduced: see also params for relevance....
;; 09.26.03: POL2SAL-strong added.
;; 12.24.03: finalPols logic in real0 was WRONG. I corrected it.
;; 12.24.03: finalPols logic: introduced back deletedPols
;; 12.24.03: lines 127,132: subMap1,2 replaced by full amaps......
;; 06.05.07: sal-abstract-expr-real-op-wrap: Removed elements
;;   from E, R, S from the witnesses...
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "hsal-abstract-real-expression")

;; REQUIRE: sal:mk-sal-equation, sal:mk-syntactic-bool, hsal-abs:RAF
;; hsal-abs:BAF, hsal-abs:DNF.
;; EXPORTS:
;;  sal-abstract-expr-real(p,tran,symtab,amap) --> (flag,c1,c2,c3)
;;  flag=t: means c1,c2,c3 are conditions under which expr> 0, = 0, < 0
;;  flag=f: means c1,c2,c3 are conditions under which expr>=0, 0, <=0
;;  tran: gives guards (invariants) which can be assumed true
;;  amap: gives polynomials over which c1, c2, c3 are constructed
;;  symtab: build-database for terms in the module

;; Local cache to store tran-->a1234 mapping.
(defvar *cache* nil)		;; sth-->(= > >= <>) conjuncts in sth.
(defvar *neg-cache* nil)	;; sth-->(= > >= <>) conjuncts in (not sth).
(defvar *p2absp-cache* nil)	;; p-->under what conditions is p >,<,=0

(defmacro print-debug (level &rest others)
  `(if (and hsal-abs:*dlevel* (> ,level hsal-abs:*dlevel*))
       (funcall #'format ,@others)))

;; ============================================================================
;; sal-abstract-expr-real: p:poly-in-new-rep transition --> c123
;; tran: provides context (guards) under which to generate c1,c2,c3.
;; symtab: provides lookup for the guards (read-only access).
;; amap: c1,c2,c3 use these polynomials (read-only, abstraction pols)
;; Return Value: Conditions ((c1.c2).c3) under which p is P, Z, N.
;; ============================================================================
(defun sal-abstract-expr-real (p tran symtab amap namap feasL E0 R0 S0)
  (declare (special hsal-abs:*dlevel*))
  (let ((ans (sal-abstract-expr-real-easy p)))
    (if ans (return-from sal-abstract-expr-real ans)))
  (let ((a1234 (pvsfml2polyrep (expression (guard tran)) symtab)))
    (multiple-value-bind (st E R S) (chaining-dp:saturate (car a1234) (cadr a1234) (caddr a1234) nil E0 R0 S0)
      (if (null st) (progn (print-debug 9 t "ERROR: Unsatisfiable guard?~%") (break)))
      (if (cadddr a1234)
	  (print-debug 9 t "WARNING: Guard is not simple!~%"))
      (chaining-dp:reset-wit E R S)		;; reset wit field in E, R, S. See *all* for why.
      (sal-abstract-expr-real0 p amap namap feasL E R S))))

(defun sal-abstract-expr-real0 (p amap namap feasL E0 R0 S0)
  (declare (special hsal-abs:*dlevel*))
  (let ((ans (sal-abstract-expr-real-easy p))) 
    (if ans (return-from sal-abstract-expr-real0 ans)))
  (let ((ans (find-if #'(lambda(x) (and (prep:polyrepEqual? (caar x) p) 
		(equal (cadar x) E0) (equal (caddar x) R0))) *p2absp-cache*))) 
    (if ans (return-from sal-abstract-expr-real0 (cdr ans))))
  (multiple-value-bind (allVars allPars) (allVarsIn p)
  (let* ((allVarsPars (append allPars allVars))
	 (E (getRelevantPols E0 allVarsPars :key #'chaining-dp::POL-pol))
	 (R (getRelevantPols R0 allVarsPars :key #'chaining-dp::POL-pol))
	 (S (getRelevantPols S0 allVarsPars :key #'chaining-dp::POL-pol))
	 (allPols (mapcar #'cdr amap))	;; all abstraction polynomials
	 (relevantPols (getRelevantPols allPols allVars))
	 (samePols (loop for i in relevantPols
			 for ans = (prep:polyrepConstMultiple? p i)
			 thereis (when ans (cons i ans)))))
    (print-debug 1 t "p2absp-cache: Not found ~A,~A,~A,~A~%" p E0 R0 S0)
    ;; (break)
    (if samePols 
	(let* ((newpol (car samePols))
	       (newans (cdr samePols))
	       (absvar (get-variable2 newpol amap)) ;; CHANGE HERE
	       (pos (sal.hsal:mk-sal-equation absvar sal.hsal:sal-pos))
	       (neg (sal.hsal:mk-sal-equation absvar sal.hsal:sal-neg))
	       (c1 (if (eq newans 1) pos neg))
	       (c2 (sal.hsal:mk-sal-equation absvar sal.hsal:sal-zero))
	       (c3 (if (eq newans 1) neg pos))
	       (ans (cons t (cons (cons c1 c2) c3))))
	  (setf *p2absp-cache* (cons (cons (list p E0 R0) ans) *p2absp-cache*))
	  ans)
	(let* ((finalPols (if (<= (length relevantPols) 3) (list relevantPols)
			      (let ((smallPols (getAllRelevantPols relevantPols allVars)))
				(cond ((eq (length smallPols) 0)
				       (list (subseq relevantPols 0 3)))
				      ((<= (length smallPols) 3)
				       (list (mapcar #'car smallPols)))
				      (t
				    (let* ((samePols (loop for i in smallPols if (cdr i) collect (car i)))
				           (restPols (loop for i in smallPols if (not (cdr i)) collect (car i)))
					   (tinyPols (remove-if-not #'prep:interval? restPols))
					   (deletedPols (set-difference restPols tinyPols)))
				      (print-debug 2 t "  restPols=~A~%" (mapcar #'prep:polyrepPrint restPols))
				      (print-debug 2 t "  tinyPols=~A~%" (mapcar #'prep:polyrepPrint tinyPols))
				      (cond ((>= (length samePols) 3)
				             (cons samePols nil))
					    ((<= (length deletedPols) 2)
				      	     (cons (append samePols deletedPols) nil))
					    (t (cons (subseq (append samePols deletedPols) 0 4) nil)))))))))
				      ;; (cons (append samePols deletedPols) tinyPols)
	       (*feasL* feasL)
	       (*namap* namap))
	  (declare (special *feasL* *namap*))
	  (let ((ans (sal-abstract-expr-real1 p (car finalPols) (cdr finalPols) amap E R S)))
	    (print-debug 4 t "ERS IS:~%E=~A~%R=~A~%S=~A~%" 
		(chaining-dp::dpPrint E) (chaining-dp::dpPrint R) (chaining-dp::dpPrint S))
	    (setf *p2absp-cache* (cons (cons (list p E0 R0) ans) *p2absp-cache*))
	    ans))))))

;; for each of the 3^|pols| cases, check if it goes in c1, c2, or c3.
(defun sal-abstract-expr-real1 (p pols1 pols2 amap ctxtE ctxtR ctxtS)
  (declare (special hsal-abs:*dlevel*))
  (let ((subMap1 (loop for i in pols1 collect (rassoc i amap)))
	(subMap2 (loop for i in pols2 collect (rassoc i amap))))
    (if pols1 (print-debug 5 t "  Searching ~A cases...~%   ~A~%" (expt 3 (length pols1)) 
			(mapcar #'prep:polyrepPrint pols1)))
    (multiple-value-bind (c1 c2 c3) (sal-abstract-expr-real-op-wrap p subMap1 ctxtE ctxtR ctxtS)
    (if pols1 (print-debug 2 t "  wrap returned ~a, ~a, ~a~%" c1 c2 c3))
    (if pols2 (print-debug 5 t "  Searching ~A cases...~%   ~A~%" (expt 3 (length pols2)) 
			(mapcar #'prep:polyrepPrint pols2)))
    (multiple-value-bind (d1 d2 d3) (sal-abstract-expr-real-op-wrap p subMap2 ctxtE ctxtR ctxtS)
    (if pols2 (print-debug 2 t "  wrap returned ~a, ~a, ~a~%" d1 d2 d3))
      (print-debug 5 t "  Searching all cases done.~%")
      (cons nil (cons (cons (sal.hsal:mk-syntactic-bool (list c1 d1) sal:'OR) 
			    (sal.hsal:mk-syntactic-bool (list c2 d2) sal:'OR)) 
		      (sal.hsal:mk-syntactic-bool (list c3 d3) sal:'OR)))))))

(defun sal-abstract-expr-real-op-wrap (p0 pols E R S)
  (declare (special *namap* hsal-abs:*dlevel*))
  (let ((p (chaining-dp::nnormalizePoly p0)))		;; IMP: since normalize doesn't keep track of witnesses!
  (print-debug 3 t "  Analyzing ~A...~%" (prep:polyrepPrint p))
  (let ((negs (loop for i in pols collect (assoc (cdr i) *namap*)))
		;; (cons (cdr i) (prep:polyrepNegativePoly (cdr i)))
	(*all* (append E R S))
	(np (prep:polyrepNegativePoly p)))
    (declare (special *all*))
    (multiple-value-bind (c1 c2 c3)
	 (if pols
	 (sal-abstract-expr-real-op p np pols negs nil nil nil nil (list (list E R S)) (list nil) (list 0)))
      (print-debug 2 t "   Before:~%    c1=~A~%    c2=~A~%    c3=~A~%" c1 c2 c3)
      (let* ((ERSp (append E R S (list p np)))
	     (nc1 (mapcar #'(lambda(x)(set-difference x ERSp :test #'myeq)) c1))
	     (nc2 (mapcar #'(lambda(x)(set-difference x ERSp :test #'myeq)) c2))
	     (nc3 (mapcar #'(lambda(x)(set-difference x ERSp :test #'myeq)) c3)))
        (print-debug 2 t "   After:~%    c1=~A~%    c2=~A~%    c3=~A~%" nc1 nc2 nc3)
	(print-debug 3 t "  Analyzing ~A done.~%" (prep:polyrepPrint p))
	(values (POLlistlist2SAL nc1 pols negs) (POLlistlist2SAL nc2 pols negs) (POLlistlist2SAL nc3 pols negs)))))))

;; x = POL, y = POL or just p
(defun myeq (x y)
  (let ((p1 (chaining-dp::POL-pol x))
	(p2 (if (chaining-dp::POL-p y) (chaining-dp::POL-pol y) y)))
    (or (prep:polyrepEqual? p1 p2)
        (eq (prep:polyrepConstMultiple? p1 p2) 1))))

;; Depth-First Search: closed--result (from the closed branches)
;; open---list of leaves in newrep-fmls, 
;; openh--list of leaves: history as (pols . op)-list
;; opend--list of depth of leaves
;; pol: polynomial-in-newrep to be guessed >= 0
;; pols: all relevant pols (i.e. 3^|pols| choices)
;; c1, c2, c3: WITNESSES; c4: all inconsistencies (redoing feasibility here!!!!), open: leaves E.R;
(defun sal-abstract-expr-real-op (pol npol pols negs c1 c2 c3 c4 ERl openh opend)
  (if (null ERl) 
      (return-from sal-abstract-expr-real-op (values c1 c2 c3)))
  (if (>= (car opend) (length pols))
      (let ((E0 (caar ERl)) (R0 (cadar ERl)) (S0 (caddar ERl)))
        (multiple-value-bind (st1 E1) (feas? npol chaining-dp::'> (car openh) c1 E0 R0 S0)
        (multiple-value-bind (st2 E2) (feas? pol chaining-dp::'= (car openh) c2 E0 R0 S0)
        (multiple-value-bind (st3 E3) (feas? pol chaining-dp::'> (car openh) c3 E0 R0 S0)
          (let ((cc1 (and (null st1) E1))
	        (cc2 (and (null st2) E2))
	        (cc3 (and (null st3) E3)))
	    ;; (if cc1 (format t "*"))
	    ;; (if cc2 (format t "*"))
	    ;; (if cc3 (format t "*"))
	    (sal-abstract-expr-real-op pol npol pols negs (if cc1 (jwrr c1 E1) c1) (if cc2 (jwrr c2 E2) c2)
		(if cc3 (jwrr c3 E3) c3) c4 (cdr ERl) (cdr openh) (cdr opend)))))))
      (let* ((q (cdr (nth (car opend) pols)))
	     (d (+ (car opend) 1))
	     (negq (cdr (assoc q negs)))
	     (E0 (caar ERl))
	     (R0 (cadar ERl))
	     (S0 (caddar ERl)))
	   (multiple-value-bind (st1 E1 R1 S1) (feas? q chaining-dp::'> (car openh) c4 E0 R0 S0 -1)
	   (multiple-value-bind (st2 E2 R2 S2) (feas? q chaining-dp::'= (car openh) c4 E0 R0 S0 -1)
	   (multiple-value-bind (st3 E3 R3 S3) (feas? negq chaining-dp::'> (car openh) c4 E0 R0 S0 q)
	     (let* ((case1 (if st1 (list (list E1 R1 S1))))
		    (case2 (if st2 (list (list E2 R2 S2))))
		    (case3 (if st3 (list (list E3 R3 S3))))
		    (hase1 (if st1 (list (acons q chaining-dp::'> (car openh)))))
		    (hase2 (if st2 (list (acons q chaining-dp::'= (car openh)))))
		    (hase3 (if st3 (list (acons negq chaining-dp::'> (car openh)))))
		    (nc4 (nconc (if (null st1) (list E1)) (if (null st2) (list E2)) (if (null st3) (list E3))))
		    (d1 (if st1 (list d)))
		    (d2 (if st2 (list d)))
		    (d3 (if st3 (list d))))
	    (sal-abstract-expr-real-op pol npol pols negs c1 c2 c3 (apply #'jwrr c4 (delete-if #'null nc4))
		    (nconc case1 case2 case3 (cdr ERl))
		    (nconc hase1 hase2 hase3 (cdr openh))
		    (nconc d1 d2 d3 (cdr opend))))))))))

;; Inspired by dp-feas::feas? function
(defun feas? (pol op poloplist infeaslist E R S &optional (npol nil))
  (declare (special *all* *feasL*))
  (let ((this (acons pol op poloplist)))
    (if (not (dp-feas:global-feas? this *feasL* npol)) (return-from feas? nil))
    (if (dp-feas:known? (acons pol op poloplist) infeaslist) nil
      	(multiple-value-bind (st E1 R1 S1) (chaining-dp:saturate nil nil nil (list pol op) E R S)
	  (if (null st) (values st (set-difference E1 *all*)) (values st E1 R1 S1))))))

;; join, while removing redundant
(defun jwrr (infeasList &rest newList)
  ; (nconc newList infeasList)
  (nconc newList (delete-if #'(lambda(x) (redundantL? x newList)) infeasList)))

(defun redundantL? (infeas infeasList)
  (some #'(lambda(x) (redundant? infeas x)) infeasList))

(defun redundant? (infeas1 infeas2)
  (subsetp infeas2 infeas1 :test #'(lambda(x y) (and
	(eq (chaining-dp::POL-pol x) (chaining-dp::POL-pol y))
	(eq (chaining-dp::POL-op x) (chaining-dp::POL-op y))))))

(defun POLlistlist2SAL (pll amap negs)
  (if (null pll) (mk-sal-nameexpr nil 'FALSE)
      (sal.hsal:mk-syntactic-bool (mapcar #'(lambda(x) (POLlist2SAL x amap negs)) pll) sal:'OR)))

(defun POLlist2SAL (pl amap negs)
  (if (null pl) (mk-sal-nameexpr nil 'TRUE)
      (sal.hsal:mk-syntactic-bool (mapcar #'(lambda(x) (POL2SAL x amap negs)) pl) sal:'AND)))

(defun POL2SAL (pol amap negs)
  (let* ((p (chaining-dp::POL-pol pol))
	 (avar1 (car (rassoc p amap)))
	 (val (if (eq (chaining-dp::POL-op pol) chaining-dp::'=) 
		  sal.hsal:sal-zero (if avar1 sal.hsal:sal-pos sal.hsal:sal-neg)))
	 (avar2 (if avar1 avar1 (car (rassoc (car (rassoc p negs)) amap)))))
    (if (null avar2) (POL2SAL-strong pol amap negs) ; (mk-sal-nameexpr nil 'TRUE) ;; break I think S not deleted from witnesses
	(sal.hsal:mk-sal-equation avar2 val))))

(defun POL2SAL-strong (pol amap negs)
  (let* ((p (chaining-dp::POL-pol pol))
	 (test #'(lambda(x y)(eq (prep:polyrepConstMultiple? x y) 1)))
	 (avar1 (car (rassoc p amap :test test)))
	 (val (if (eq (chaining-dp::POL-op pol) chaining-dp::'=) 
		  sal.hsal:sal-zero (if avar1 sal.hsal:sal-pos sal.hsal:sal-neg)))
	 (avar2 (if avar1 avar1 (car (rassoc (car (rassoc p negs :test test)) amap)))))
    (if (null avar2) (break)) ; (mk-sal-nameexpr nil 'TRUE) ;; break I think S not deleted from witnesses
    (sal.hsal:mk-sal-equation avar2 val)))

(defun sal-abstract-expr-real-easy (p)
  (if (or (null p) (and (null (cdr p)) (null (cdar p))))
      (let* ((tt (sal:mk-sal-nameexpr nil 'TRUE))
	     (ff (sal:mk-sal-nameexpr nil 'FALSE)))
	(cond ((null p) 
	       (cons t (cons (cons ff tt) ff)))
	      ((> (caar p) 0)
	       (cons t (cons (cons tt ff) ff)))
	      (t 
	       (cons t (cons (cons ff ff) tt)))))))
;; ============================================================================

;; ============================================================================
;; Input:  a sal:fml
;; Output: (a1 a2 a3 a4) s.t. input = (a1=0) AND (a2>0) AND (a3>=0) AND a4
;; Output: Equations, >, >=, <>. REMOVED REST.
;; Eg. Input: x=4 AND (x=4 OR y>5) Output: ((x-4), nil, nil, (x=4 OR y>5))
;; Output: in RAF.
;; ============================================================================
(defun pvsfml2polyrep (fml symtab)
  (declare (special hsal-abs:*dlevel*))
  (let ((a1234 (cdr (assoc fml *cache*))))		;; Cache lookup first!
    (if a1234 a1234
	(let* ((a1234 (separate-real-conjuncts* fml symtab)))
	  (print-debug 1 t "fml2prep-cache: Not found ~A~%" fml)
	  (setf *cache* (acons fml a1234 *cache*))	;; update *cache*
	  a1234))))

(defmethod separate-real-conjuncts* ((fml hsal-abs:BAF) symtab)
  (declare (ignore symtab))
  (list nil nil nil nil))

(defmethod separate-real-conjuncts* ((fml hsal-abs:RAF) symtab)
  (declare (ignore symtab))
  (cond ((eq (hsal-abs:RAF-op fml) '=)
	 (list (list (hsal-abs:RAF-pol fml)) nil nil nil))
	((eq (hsal-abs:RAF-op fml) '>)
	 (list nil (list (hsal-abs:RAF-pol fml)) nil nil))
	((eq (hsal-abs:RAF-op fml) '<)
	 (list nil (list (prep:polyrepNegativePoly (hsal-abs:RAF-pol fml))) nil nil))
	((eq (hsal-abs:RAF-op fml) '>=)
	 (list nil nil (list (hsal-abs:RAF-pol fml)) nil))
	((eq (hsal-abs:RAF-op fml) '<=)
	 (list nil nil (list (prep:polyrepNegativePoly (hsal-abs:RAF-pol fml))) nil))
	((eq (hsal-abs:RAF-op fml) '<>)		;; FUTURE: CHECK
	 (list nil nil nil (list (hsal-abs:RAF-pol fml))))
	(t (sal-error fml "Cant handle~%"))))

(defmethod separate-real-conjuncts* ((fml hsal-abs:DNF) symtab)
  (if (cdr (hsal-abs:DNF-products fml)) (list nil nil nil nil)
      (let* ((all (loop for i in (car (hsal-abs:DNF-products fml)) collect
			(separate-real-conjuncts* i symtab)))
	     (all1 (loop for i in all nconc (car i)))
	     (all2 (loop for i in all nconc (cadr i)))
	     (all3 (loop for i in all nconc (caddr i)))
	     (all4 (loop for i in all nconc (cadddr i))))
	(list all1 all2 all3 all4))))

(defmethod separate-real-conjuncts* ((fml t) symtab)
  (let* ((expr1 (cdr (assoc fml symtab :test #'eq))))
    (if (not expr1) (progn (format t "Incorrect database:~A~%" fml) (break)))
    (separate-real-conjuncts* expr1 symtab)))
;; ============================================================================

;; ============================================================================
;; Same as above, but do it for the negation of the formula.
;; ============================================================================
(defun pvsfml2polyrepneg (fml symtab)
  (declare (special hsal-abs:*dlevel*))
  (let ((a1234 (cdr (assoc fml *neg-cache*))))		;; Cache lookup first!
    (if a1234 a1234
	(let* ((a1234 (separate-real-conjuncts-neg* fml symtab)))
	  (print-debug 1 t "fml2prepneg-cache: Not found ~A~%" fml)
	  (setf *neg-cache* (acons fml a1234 *neg-cache*))	;; update *cache*
	  a1234))))

(defmethod separate-real-conjuncts-neg* ((fml hsal-abs:BAF) symtab)
  (declare (ignore symtab))
  (list nil nil nil nil))

(defmethod separate-real-conjuncts-neg* ((fml hsal-abs:RAF) symtab)
  (declare (ignore symtab))
  (cond ((eq (hsal-abs:RAF-op fml) '<>)
	 (list (list (hsal-abs:RAF-pol fml)) nil nil nil))
	((eq (hsal-abs:RAF-op fml) '<=)
	 (list nil (list (hsal-abs:RAF-pol fml)) nil nil))
	((eq (hsal-abs:RAF-op fml) '>=)
	 (list nil (list (prep:polyrepNegativePoly (hsal-abs:RAF-pol fml))) nil nil))
	((eq (hsal-abs:RAF-op fml) '<)
	 (list nil nil (list (hsal-abs:RAF-pol fml)) nil))
	((eq (hsal-abs:RAF-op fml) '>)
	 (list nil nil (list (prep:polyrepNegativePoly (hsal-abs:RAF-pol fml))) nil))
	((eq (hsal-abs:RAF-op fml) '=)		;; FUTURE: CHECK
	 (list nil nil nil (list (hsal-abs:RAF-pol fml))))
	(t (sal-error fml "Cant handle~%"))))

(defmethod separate-real-conjuncts-neg* ((fml hsal-abs:DNF) symtab)
  (let* ((all (loop for i in (hsal-abs:DNF-products fml) collect
			(if (cdr i) (list nil nil nil nil)
				    (separate-real-conjuncts-neg* (car i) symtab))))
	 (all1 (loop for i in all nconc (car i)))
	 (all2 (loop for i in all nconc (cadr i)))
	 (all3 (loop for i in all nconc (caddr i)))
	 (all4 (loop for i in all nconc (cadddr i))))
    (list all1 all2 all3 all4)))

(defmethod separate-real-conjuncts-neg* ((fml t) symtab)
  (let* ((expr1 (cdr (assoc fml symtab :test #'eq))))
    (if (not expr1) (progn (format t "Incorrect database:~A~%" fml) (break)))
    (separate-real-conjuncts-neg* expr1 symtab)))
;; ============================================================================

;; ============================================================================
;; Local copy of get-variable2.
;; ============================================================================
(defun get-variable2 (p amap)
  (let ((ans (car (rassoc p amap))))
    (if ans ans (car (rassoc p amap :test #'prep:polyrepEqual?)))))
;; ============================================================================
