;; --------------------------------------------------------------------
;; HybridSAL
;; Copyright (C) 2006, SRI International.  All Rights Reserved.
;; 
;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License
;; as published by the Free Software Foundation; either version 2
;; of the License, or (at your option) any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
;; --------------------------------------------------------------------

;;;;;;;;;;;;;;;;;;;;;;;;;;* -*- Mode: Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; vim: syntax=lisp
;; hsal-abs-bool.lisp --
;; Author          : Ashish Tiwari
;; Created On      : Sat Oct 19, 2002
;; Last Modified By: Ashish Tiwari
;; Last Modified On: Sat Oct 19, 2002
;; Update Count    : 0
;; Status          : Unknown, use with caution
;;
;; HISTORY : 
;; 10.19.02: Abstracting (Boolean) Atomic Formulas over Reals!
;; 11.20.02: DOES NOT USE INVARIANT information!!!!
;; 11.11.10: Introduced sal-abstract-fmla-conj to abstract conj at once
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "hsal-abstractor-formula")

;; REQUIRES: sal:mk-syntactic-bool*,
;; prep:polyrepNegativePoly, hsal-abs:RAFetc
;; chaining-dp: decideDynamicER.
;; EXPORTS:
;;  sal-abstract-formula(fmla,db,symtab,amap) --> new-fmla and new-db
;;  symtab: is the database of terms constructed by build-database (to lookup fmla)
;;  db: is the database of already abstracted fmlas
;;  fmla: RAF/BAF/DNF/rest.
;;  amap: the abstraction map!

;; ============================================================================
;; Debug print-debug 2 option.
;; ============================================================================
(defmacro print-debug (level &rest others)
  `(if (and hsal-abs:*dlevel* (> ,level hsal-abs:*dlevel*))
       (funcall #'format ,@others)))
;; ============================================================================

;; ============================================================================
;; sal-abstract-formula: DNF-newfml db-of-old-results --> DNF-newfml new-db
;; symtab: mapping for converting sal:fmls to BAF/RAF/DNF.
;; Return Values: abstract-formula, new-database, cont-component in input fmla
;; ============================================================================
(defun sal-abstract-formula (fmla db symtab amap)
  (declare (special hsal-abs:*dlevel*))
  (print-debug 5 t " Abstracting formula ") (print-debug 5 t "" (sal:unparse fmla))
  (print-debug 5 t "~%")
  (multiple-value-bind (afmla ndb) (sal-abstract-formula* fmla db symtab amap)
    ;; (print-debug 5 t " Abstracting formula  returns ~a~%" afmla)
    (print-debug 5 t " Abstracting formula ... done.~%")
    ;; (print-debug 5 t " Abstracting formula ... amap = ~%~a.~%" amap)
    ;; (break)
    (values afmla ndb)))

(defun sal-abstract-formula* (fmla db symtab amap)
  (cond ((BAF-p fmla)
	 (values (baf2salfmla fmla) db))	;; (baf2salfmla fmla))
	((RAF-p fmla)
    	 (cond ((member (RAF-op fmla) '(> < = >= <= /=))
		(check-and-update-database (RAF-pol fmla) (RAF-op fmla) db amap))
	       ;((eq (RAF-op fmla) '<=)
		;(check-and-update-database2 (RAF-pol fmla) '< '= db amap))
	       ;((eq (RAF-op fmla) '>=)
		;(check-and-update-database2 (RAF-pol fmla) '> '= db amap))
	       (t ;; (eq (RAF-op fmla) '<>)
	 	(sal-error t "PANIC: Unknown OPERATOR ~A~%" (RAF-op fmla)) (break))))
		;(check-and-update-database2 (RAF-pol fmla) '> '< db amap)
	((DNF-p fmla)
	 (multiple-value-bind (fmls ndb)
		(sal-abstract-fmla-list2* (DNF-products fmla) db symtab amap)
	   (values (make-DNF :products fmls) ndb)))
	((sal:expression? fmla)
	 (sal-abstract-formula* (cdr (assoc fmla symtab)) db symtab amap))
	(t
	 (sal-error t "PANIC: Unknown formula type ~A" fmla))))

(defun sal-abstract-fmla-list2* (llist db symtab amap &optional (res nil))
  (if (null llist)
      (values (nreverse res) db)
      (multiple-value-bind (f1 db1) 
	;; (sal-abstract-fmla-list1* (car llist) db symtab amap)
	(sal-abstract-fmla-conj (car llist) db symtab amap)
        (sal-abstract-fmla-list2* (cdr llist) db1 symtab amap (cons f1 res)))))

(defun sal-abstract-fmla-list1* (llist db symtab amap &optional (res nil))
  (if (null llist)
      (values (nreverse res) db)
      (multiple-value-bind (f1 db1) 
	(sal-abstract-formula* (car llist) db symtab amap)
        (sal-abstract-fmla-list1* (cdr llist) db1 symtab amap (cons f1 res)))))
;; ============================================================================

;; ============================================================================
;; abstracting a conjunction of formulas all together; not separately
(defun sal-abstract-fmla-conj (flist db symtab amap)
  (multiple-value-bind (rafs others) 
	(partition-rafs-others flist symtab)
    (multiple-value-bind (abs-others db1) 
	(sal-abstract-fmla-list1* others db symtab amap)
      (values (nconc abs-others (sal-abstract-fmla-conj* rafs amap)) db1))))

(defun partition-rafs-others (flist symtab &optional (rafs nil) (others nil))
  (if (null flist) (values rafs others)
      (let* ((f1 (car flist))
	     (f2 (if (sal:expression? f1) (cdr (assoc f1 symtab)) f1)))
        (if (RAF-p f2)
            (partition-rafs-others (cdr flist) symtab (cons f2 rafs) others)
            (partition-rafs-others (cdr flist) symtab rafs (cons f1 others))))))
  
;; flist = list of RAFs that need to be abstracted...
(defun sal-abstract-fmla-conj* (flist amap)
  (multiple-value-bind (E R S) (rafs2ERS flist)
    (multiple-value-bind (st E1 R1 S1) (chaining-dp:saturate E R S)
      (when (null st)
	(sal-error t "PANIC: Formula to be abstracted is UNSATISFIABLE~%") 
	(break))
      (sal-abstract-up-new* E1 R1 S1 amap nil))))

(defun rafs2ERS (flist &optional (E nil) (R nil) (S nil))
  (if (null flist) (values E R S)
      (let ((pol (RAF-pol (car flist))))
        (case (RAF-op (car flist))
	  (> (rafs2ERS (cdr flist) E (cons pol R) S))
	  (< (rafs2ERS (cdr flist) E (cons (prep:polyrepNegativePoly pol) R) S))
	  (= (rafs2ERS (cdr flist) (cons pol E) R S))
	  (>= (rafs2ERS (cdr flist) E R (cons pol S)))
	  (<= (rafs2ERS (cdr flist) E R (cons (prep:polyrepNegativePoly pol) S)))
	  (t (sal-error t "PANIC: Unknown OPERATOR ~A" (RAF-op (car flist))) (break))))))
;; ============================================================================

;; ============================================================================
;; database.lisp
;; Database is ((db1 . db2) . db3), where dbi is a list of (fmla . afmla)
;; ============================================================================
(defun check-and-update-database2 (fmla op1 op2 db amap)
  (multiple-value-bind (afml1 db1) 
	(check-and-update-database fmla op1 db amap)
  (multiple-value-bind (afml2 db2) 
	(check-and-update-database fmla op2 db1 amap)
    (values (sal.hsal:mk-syntactic-bool* (list afml1 afml2) sal:'OR) db2))))

(defun check-and-update-database (fmla op db amap)
  (declare (special hsal-abs:*dlevel*))
  (let ((afmla (check-database fmla op db)))
    (if afmla (values afmla db)
	(let* ((afmla (sal-abstract-formula-ext fmla op amap))
	       (db1 (update-database fmla op afmla db)))
	  (print-debug 2 t " fmla ~A ~A 0 not found in database~%" fmla op)
	  (values afmla db1)))))

(defun check-database (fmla op0 db)
  (let* ((fn #'(lambda (x y) (and (eq (RAF-pol x) (RAF-pol y))
				  (eq (RAF-op x) (RAF-op y)))))
	 (fml (make-RAF :pol fmla :op op0))
	 (afml (cdr (assoc fml db :test fn))))
    afml))

(defun update-database (fmla op afmla db)
 (acons (make-RAF :pol fmla :op op) afmla db))
;; ============================================================================

;; ============================================================================
;; decision-procedure.lisp
;; ============================================================================
(defun sal-abstract-formula-ext (pol op amap)
  (let* ((pol1 (if (or (eq op '<) (eq op '<=)) (prep:polyrepNegativePoly pol) pol))
	 (op1  (if (eq op '<) '> (if (eq op '<=) '>= op)))
         (fml2 (sal-abstract-up-new pol1 op1 amap)))
    (sal.hsal:mk-syntactic-bool* fml2 sal:'AND)))

(defun sal-abstract-up-new (fml1 op1 amap)
  (let ((ans (sal-abstract-up-new-easy fml1 op1 amap)))
    (if ans (return-from sal-abstract-up-new ans)))
  (multiple-value-bind (status E1 R1 S1)
	(case op1
	  (> (chaining-dp:decideDynamicR fml1 nil nil nil)) ;; CHECK
	  (>= (chaining-dp:decideDynamicS fml1 nil nil nil)) ;; CHECK
	  (= (chaining-dp:decideDynamicE fml1 nil nil nil)) ;; CHECK
	  (t (sal-error t "PANIC: Unknown OPERATOR ~A" op1) (break)))
    (declare (ignore status))
    (sal-abstract-up-new* E1 R1 S1 amap nil)))
 
(defun sal-abstract-up-new-easy (fml1 op amap)
  (let* ((ans1 (rassoc fml1 amap))
	 (ans2 (if ans1 (cons ans1 1)
		   (loop for i in amap
			 for j = (prep:polyrepConstMultiple? fml1 (cdr i))
			 thereis (when j (cons i j))))))
    (if (null ans2) (return-from sal-abstract-up-new-easy nil))
    (let* ((sign (eq (cdr ans2) 1))
	   (avar (caar ans2))
	   (nop (if sign op (case op (> '<) (< '>) (>= '<=) (<= '>=) (t op))))
	   (aval (case nop (> sal.hsal:sal-pos) (< sal.hsal:sal-neg) 
			   (= sal.hsal:sal-zero) (t nil)))
	   (aval1 (case nop (>= sal.hsal:sal-neg) (<= sal.hsal:sal-pos) (t sal.hsal:sal-zero))))
      (if aval 
	  (list (sal.hsal:mk-sal-equation avar aval))
	  (list (sal.hsal:mk-sal-disequation avar aval1))))))

(defun sal-abstract-up-new* (E1 R1 S1 amap res)
  (if (null amap) res
      (let ((absvar (caar amap))
	    (polrep (cdar amap)))
	(multiple-value-bind (A1 A2 A3) 
		(check-implication-cases E1 R1 S1 polrep)
	  (cond ((and A2 A3)
	    	 (sal-abstract-up-new* E1 R1 S1 (cdr amap)
			(cons (sal.hsal:mk-sal-equation absvar sal.hsal:sal-pos) res)))
		((and A1 A2)
		 (sal-abstract-up-new* E1 R1 S1 (cdr amap)
			(cons (sal.hsal:mk-sal-equation absvar sal.hsal:sal-neg) res)))
		((and A1 A3)
		 (sal-abstract-up-new* E1 R1 S1 (cdr amap)
			(cons (sal.hsal:mk-sal-equation absvar sal.hsal:sal-zero) res)))
		(A1
		 (sal-abstract-up-new* E1 R1 S1 (cdr amap)
			(cons (sal.hsal:mk-sal-disequation absvar sal.hsal:sal-pos) res)))
		(A2
		 (sal-abstract-up-new* E1 R1 S1 (cdr amap)
			(cons (sal.hsal:mk-sal-disequation absvar sal.hsal:sal-zero) res)))
		(A3
		 (sal-abstract-up-new* E1 R1 S1 (cdr amap)
			(cons (sal.hsal:mk-sal-disequation absvar sal.hsal:sal-neg) res)))
		(t 
		 (sal-abstract-up-new* E1 R1 S1 (cdr amap) res)))))))

(defun check-implication-cases (E1 R1 S1 polyrep)
  (let ((s2 (chaining-dp:decideDynamicR polyrep E1 R1 S1))
	(s3 (chaining-dp:decideDynamicE polyrep E1 R1 S1))
	(s4 (chaining-dp:decideDynamicR 
		(prep:polyrepNegativePoly polyrep) E1 R1 S1)))
    (values (eq s2 chaining-dp:'inconsistent) 
		(eq s3 chaining-dp:'inconsistent) 
		(eq s4 chaining-dp:'inconsistent))))

(defun baf2salfmla (baf)
  (let* ((var (mk-sal-nameexpr nil (BAF-var baf)))
	 (fml (sal.hsal:mk-sal-equation var (BAF-val baf))))
    (if (BAF-sgn baf) fml
	(mk-sal-application nil (mk-sal-nameexpr nil 'NOT) (mk-sal-tupleliteral nil fml)))))
;; ============================================================================

