;; --------------------------------------------------------------------
;; HybridSAL
;; Copyright (C) 2006, SRI International.  All Rights Reserved.
;; 
;; This program is free software; you can redistribute it and/or
;; modify it under the terms of the GNU General Public License
;; as published by the Free Software Foundation; either version 2
;; of the License, or (at your option) any later version.
;; 
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;; 
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
;; --------------------------------------------------------------------

;;;;;;;;;;;;;;;;;;;;;;;;;;* -*- Mode: Lisp -*- ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; vim: syntax=lisp
;; hsal-abs-db.lisp --
;; Author          : Ashish Tiwari
;; Created On      : Fri Oct 18, 2002
;; Last Modified By: Ashish Tiwari
;; Last Modified On: Fri Oct 18, 2002
;; Update Count    : 0
;; Status          : Unknown, use with caution
;;
;; HISTORY : 
;; 10.18.02: Database of expressions in SAL spec. CommonSubExpression.
;; 10.19.02: Invar-decl included in database.
;; 11.05.02: RAF also used for storing polynomials (in conditional exprs).
;; 12.05.02: Added definition: into build-database
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "hsal-abs-db")

;; REQUIRES: sal:*, RAF, etc. PVS --> hsal-Representation
;; pvs2prep:pvs2polyrep.
;; Exports: 
;;  build-database(module,db0) --> new database (of all terms in module)
;;  seed-database(pols,db0) --> new database with pols added
;;  getRAFsFromDB(stack,db) --> return all RAFs corr to exprs in stack in db.

(defmacro print-debug (level &rest others)
  `(if (and hsal-abs:*dlevel* (> ,level hsal-abs:*dlevel*))
       (funcall #'format ,@others)))

;; ============================================================================
;; build-database: run down base-module and construct symtab like thing.
;; db: {SAL-stuff,sal:expr} --> DNF over {sal:expr,(nrep,op)}
;; ============================================================================
(defun build-database (module &optional (db nil))
  (let* ((decls (declarations module))
	 (initfmla (find-if #'initfor-decl? decls))
	 (init (find-if #'init-decl? decls))
	 (tdecl (find-if #'trans-decl? decls))
	 (invariant (find-if #'invar-decl? decls)) 
	 (def-decl (find-if #'def-decl? decls))
	 (db1 (if def-decl (get-database-expr-list (sal.hsal:get-rhs-exprs (definitions def-decl)) db) db))
	 (db2 (if initfmla (get-database-expr (expression initfmla) db1) db1))
	 (db3 (get-database-defs-cmds init db2))
	 (db4 (get-database-defs-cmds tdecl db3))
	 (db5 (if invariant (get-database-expr (expression invariant) db4) db4)))
    ;(break)
    db5))

(defun get-database-expr (expr db)
  (get-database-expr* expr expr db))

(defun get-database-expr* (sal-expr pvs-expr db)
  (declare (special hsal-abs:*dlevel*))
  (cond ((sal:real-atomic-fmla? pvs-expr)
	 (let ((new-expr (lookup-real-expr pvs-expr db)))
	   (acons sal-expr new-expr db)))
	((sal:bool-atomic-fmla? pvs-expr)
	 (let ((var (sal:baf2boolvar pvs-expr))
	       (val (sal:baf2boolval pvs-expr))
	       (sgn (sal:baf2boolsgn pvs-expr)))
	   (acons sal-expr (hsal-abs:make-BAF :var var :val val :sgn sgn) db)))
	((or (sal:conditional? pvs-expr) (sal:chained-conditional? pvs-expr))
	 (let* ((c1 (nth 0 (sal:exprs (sal:argument pvs-expr))))
		(v1 (nth 1 (sal:exprs (sal:argument pvs-expr))))
		(v2 (nth 2 (sal:exprs (sal:argument pvs-expr))))
		(conds (list c1 v1 v2))
		(db1 (get-database-list1* conds db)))
	   (acons sal-expr (hsal-abs:make-ITE :c c1 :v1 v1 :v2 v2) db1)))
	((sal:tc-eq pvs-expr 'sal::BOOLEAN)
      	 (let* ((e1d (sal:disjuncts pvs-expr))
		;; pvs3.0/src/prover/strategies.lisp
		(e1c (loop for i in e1d collect (sal:conjuncts i)))
	   	(db1 (get-database-list2* e1c db)))
	   (acons sal-expr (hsal-abs:make-DNF :products e1c) db1)))
	((sal:tc-eq pvs-expr 'sal::REAL)
	 (let ((new-expr (lookup-real-expr1 pvs-expr db)))
	   (acons sal-expr new-expr db)))
	(t (print-debug 9 t "Error: Unknown expression~%") (break) db)))

(defun get-database-list2* (ell db)
  (if (null ell) db
      (let* ((db1 (get-database-list1* (car ell) db)))
	(get-database-list2* (cdr ell) db1))))

(defun get-database-list1* (el db)
  (if (null el) db
      (let* ((db1 (get-database-expr* (car el) (car el) db)))
	(get-database-list1* (cdr el) db1))))

;; expr:real-atomic-pvs-expr  return:(nrep.op) either from db/new
(defun lookup-real-expr (expr db)
  (let* ((neg (sal::negation? expr))
	 (expr1 (if neg (sal:args1 expr) expr))
	 (op (sal:op2symbol (sal:operator expr1)))
	 (nop (if neg (negate-op op) op))
	 (p1 (sal:args1 expr1))
	 (p2 (sal:args2 expr1))
	 (np1 (pvs2prep:pvs2polyrep p1))
	 (np2 (pvs2prep:pvs2polyrep p2))
	 (np (prep:polyrepAddPoly np1 (prep:polyrepNegativePoly np2)))
    	 (nrepop (lookup-database* np db))
	 (newop (if (eq (cdr nrepop) 1) nop (minus-op nop))))
    ;(break)
    (if nrepop (hsal-abs:make-RAF :pol (car nrepop) :op newop) 
	       (hsal-abs:make-RAF :pol np :op nop))))

(defun lookup-real-expr1 (expr db)
  (let* ((np (pvs2prep:pvs2polyrep expr))	;; expr = 0 => np = nil
    	 (nrep (if np (lookup-database1* np db) nil)))
    (if nrep (hsal-abs:make-RAF :pol nrep :op 1)
	     (hsal-abs:make-RAF :pol np :op 1))))

(defun lookup-database* (nrep db)
  (loop for i in db when 
	(if (hsal-abs:RAF-p (cdr i)) 
	    (let ((ans (prep:polyrepConstMultiple? nrep 
			(hsal-abs:RAF-pol (cdr i)))))
	      (if ans (cons (hsal-abs:RAF-pol (cdr i)) ans)))) return it))

(defun lookup-database1* (nrep db)
  (loop for i in db when 
	(if (hsal-abs:RAF-p (cdr i)) 
	    (let ((ans (prep:polyrepEqual? nrep 
			(hsal-abs:RAF-pol (cdr i)))))
	      (if ans (hsal-abs:RAF-pol (cdr i))))) return it))

(defun get-database-defs-cmds (tdecl db)
  (declare (special hsal-abs:*dlevel*))
  (if (null tdecl) db
  (let* ((cmds (definitions-or-commands tdecl))
	 (guards (cond ((sal.hsal:piecewise-continuous? cmds)
			(sal.hsal:get-rhs-exprs cmds))
		       ((sal.hsal:hybrid-automaton? cmds)
			(sal.hsal:get-guards cmds))
		       ((every #'sal:guarded-command? cmds)
			(sal.hsal:get-guards cmds))
		       (t 	;; init-decl, or, definitions
			(print-debug 2 t "Transition type unindentified: CHECK .~%")
			nil))))
    (get-database-expr-list guards db))))
 
(defun get-database-expr-list (exprlist db)
  (if (null exprlist) db
      (get-database-expr-list (cdr exprlist) 
	(get-database-expr (car exprlist) db))))

(defun seed-database (pvs-pols &optional (db nil))
  (if (null pvs-pols) db
      (let* ((nrep-pol (pvs2prep:pvs2polyrep (car pvs-pols))))
	(if (rassoc nrep-pol db :key #'hsal-abs:RAF-pol 
		:test #'prep:polyrepConstMultiple?)
	    (seed-database (cdr pvs-pols) db)
	    (seed-database (cdr pvs-pols) (acons (car pvs-pols)
		(hsal-abs:make-RAF :pol nrep-pol :op '?) db))))))

(defun getRAFsFromDB (stack db &optional (res nil))
  (if (null stack) res
      (let ((this (cdr (assoc (car stack) db))))
	(cond ((hsal-abs:RAF-p this)
	       (getRAFsFromDB (cdr stack) db (cons (hsal-abs:RAF-pol this) res)))
	      ((hsal-abs:BAF-p this)
	       (getRAFsFromDB (cdr stack) db res))
	      ((hsal-abs:DNF-p this)
	       (getRAFsFromDB (append (cdr stack) 
				      (apply #'append (hsal-abs:DNF-products this)))
		db res))
	      ((hsal-abs:ITE-p this)
	       (getRAFsFromDB (append (list (hsal-abs:ITE-c this) (hsal-abs:ITE-v1 this) 
					(hsal-abs:ITE-v2 this)) (cdr stack)) db res))
	      (t ;(format t "ERROR?: Incorrect database.~%") (break)
		 (getRAFsFromDB (cdr stack) db res))))))

(defun negate-op (op)
  (case op (< '>=) (> '<=) (= '/=) (>= '<) (<= '>) (/= '=) 
	   (t (sal-error op "Unidentified operator~A" op))))

(defun minus-op (op)
  (case op (< '>) (> '<) (= '=) (>= '<=) (<= '>=) (/= '/=) 
	   (t (sal-error op "Unidentified operator~A" op))))

;; ==========================================================
;; Functions on Boolean Atomic Formulas.
;; ==========================================================
;; ============================================================================

